"use strict";
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
class StandardServerBanSynchronisationCapabilityRenderer {
    constructor(description, messageCollector, capability) {
        this.description = description;
        this.messageCollector = messageCollector;
        this.capability = capability;
        this.requiredEventPermissions = this.capability.requiredEventPermissions;
        this.requiredPermissions = this.capability.requiredPermissions;
        this.requiredStatePermissions = this.capability.requiredStatePermissions;
        // nothing to do.
    }
    async outcomeFromIntentInRoom(roomID, projection) {
        const capabilityResult = await this.capability.outcomeFromIntentInRoom(roomID, projection);
        const title = (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            "Setting server ACL in ",
            matrix_basic_types_1.Permalinks.forRoom(roomID),
            " as it is out of sync with watched policies."));
        // only add the message if we failed, otherwise it's too spammy.
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        return capabilityResult;
    }
    async outcomeFromIntentInRoomSet(projection) {
        const capabilityResult = await this.capability.outcomeFromIntentInRoomSet(projection);
        const title = interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, "Updating server ACL in protected rooms.");
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        // Only show this when results are failing.
        if (!capabilityResult.ok.isEveryResultOk) {
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderRoomSetResult)(capabilityResult.ok, {
                summary: (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, this.description.name),
                    ": ",
                    title)),
                showOnlyFailed: true,
            }));
        }
        return capabilityResult;
    }
}
(0, matrix_protection_suite_1.describeCapabilityRenderer)({
    name: "ServerACLSynchronisationCapability",
    description: "Render the server ban capability.",
    interface: "ServerBanSynchronisationCapability",
    factory(description, draupnir, capability) {
        return new StandardServerBanSynchronisationCapabilityRenderer(description, draupnir.capabilityMessageRenderer, capability);
    },
    isDefaultForInterface: true,
});
(0, matrix_protection_suite_1.describeCapabilityContextGlue)({
    name: "ServerACLSynchronisationCapability",
    glueMethod: function (protectionDescription, draupnir, capabilityProvider) {
        return capabilityProvider.factory(protectionDescription, {
            stateEventSender: draupnir.clientPlatform.toRoomStateEventSender(),
            protectedRoomsSet: draupnir.protectedRoomsSet,
        });
    },
});
(0, matrix_protection_suite_1.describeCapabilityContextGlue)({
    name: "SimulatedServerBanSynchronisationCapability",
    glueMethod: function (protectionDescription, draupnir, capabilityProvider) {
        return capabilityProvider.factory(protectionDescription, {
            protectedRoomsSet: draupnir.protectedRoomsSet,
        });
    },
});
//# sourceMappingURL=ServerACLConsequencesRenderer.js.map