"use strict";
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
// yeah i know this is a bit insane but whatever, it can be our secret.
function renderResultForUserInSetMap(usersInSetMap, { ingword, nnedword, description, }) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, description.name),
            ": ",
            ingword,
            " ",
            usersInSetMap.map.size,
            " ",
            " ",
            usersInSetMap.map.size === 1 ? "user" : "users",
            " from protected rooms -  ",
            (0, mps_interface_adaptor_1.renderOutcome)(usersInSetMap.isEveryResultOk),
            "."),
        [...usersInSetMap.map.entries()].map(([userID, roomResults]) => renderRoomSetResultForUser(roomResults, userID, nnedword, {}))));
}
function renderRoomSetResultForUser(roomResults, userID, nnedword, { description }) {
    return (0, mps_interface_adaptor_1.renderRoomSetResult)(roomResults, {
        summary: (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            description === undefined ? ("") : (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, description.name),
                ":")),
            userID,
            " will be ",
            nnedword,
            " from ",
            roomResults.map.size,
            " rooms -  ",
            (0, mps_interface_adaptor_1.renderOutcome)(roomResults.isEveryResultOk),
            ".")),
    });
}
class StandardUserConsequencesRenderer {
    constructor(description, messageCollector, capability) {
        this.description = description;
        this.messageCollector = messageCollector;
        this.capability = capability;
        this.requiredEventPermissions = this.capability.requiredEventPermissions;
        this.requiredPermissions = this.capability.requiredPermissions;
        this.requiredStatePermissions = this.capability.requiredStatePermissions;
        // nothing to do.
    }
    async consequenceForUserInRoom(roomID, userID, reason) {
        const capabilityResult = await this.capability.consequenceForUserInRoom(roomID, userID, reason);
        const title = (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
            "Banning user ",
            userID,
            " in ",
            matrix_basic_types_1.Permalinks.forRoom(roomID),
            " for ",
            reason,
            "."));
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        this.messageCollector.addOneliner(this.description, this.capability, title);
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
    async consequenceForUsersInRoomSet(targets) {
        const capabilityResult = await this.capability.consequenceForUsersInRoomSet(targets);
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            const title = (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, "Applying policy revision to protected rooms"));
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        const usersInSetMap = capabilityResult.ok;
        if (usersInSetMap.map.size === 0) {
            return capabilityResult;
        }
        this.messageCollector.addMessage(this.description, this.capability, renderResultForUserInSetMap(usersInSetMap, {
            ingword: "Banning",
            nnedword: "banned",
            description: this.description,
        }));
        return capabilityResult;
    }
    async consequenceForUsersInRoom(roomID, targets) {
        const capabilityResult = await this.capability.consequenceForUsersInRoom(roomID, targets);
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            const title = (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                "Applying policy revision to",
                " ",
                (0, mps_interface_adaptor_1.renderRoomPill)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID, []))));
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        const resultMap = capabilityResult.ok;
        if (resultMap.map.size === 0) {
            return capabilityResult;
        }
        this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderResultForUsersInRoom)(resultMap, {
            summary: (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                    interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, this.description.name),
                    ":"),
                resultMap.map.size,
                " will be banned from",
                " ",
                (0, mps_interface_adaptor_1.renderRoomPill)(matrix_basic_types_1.MatrixRoomReference.fromRoomID(roomID)),
                " -  ",
                (0, mps_interface_adaptor_1.renderOutcome)(resultMap.isEveryResultOk),
                ".")),
        }));
        return capabilityResult;
    }
    async unbanUserFromRoomSet(userID, reason) {
        const capabilityResult = await this.capability.unbanUserFromRoomSet(userID, reason);
        if ((0, matrix_protection_suite_1.isError)(capabilityResult)) {
            const title = (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
                "Unbanning ",
                userID,
                " from protected rooms"));
            this.messageCollector.addMessage(this.description, this.capability, (0, mps_interface_adaptor_1.renderFailedSingularConsequence)(this.description, title, capabilityResult.error));
            return capabilityResult;
        }
        const usersInSetMap = capabilityResult.ok;
        if (usersInSetMap.map.size === 0) {
            return capabilityResult;
        }
        this.messageCollector.addMessage(this.description, this.capability, renderRoomSetResultForUser(usersInSetMap, userID, "unbanned", {
            description: this.description,
        }));
        return capabilityResult;
    }
}
(0, matrix_protection_suite_1.describeCapabilityRenderer)({
    name: "StandardUserConsequences",
    description: "Renders your mum uselesss",
    interface: "UserConsequences",
    factory(description, draupnir, capability) {
        return new StandardUserConsequencesRenderer(description, draupnir.capabilityMessageRenderer, capability);
    },
    isDefaultForInterface: true,
});
(0, matrix_protection_suite_1.describeCapabilityContextGlue)({
    name: "StandardUserConsequences",
    glueMethod: function (protectionDescription, draupnir, capabilityProvider) {
        return capabilityProvider.factory(protectionDescription, {
            roomBanner: draupnir.clientPlatform.toRoomBanner(),
            roomUnbanner: draupnir.clientPlatform.toRoomUnbanner(),
            setMembership: draupnir.protectedRoomsSet.setRoomMembership,
        });
    },
});
(0, matrix_protection_suite_1.describeCapabilityContextGlue)({
    name: "SimulatedUserConsequences",
    glueMethod: function (protectionDescription, draupnir, capabilityProvider) {
        return capabilityProvider.factory(protectionDescription, {
            setMembership: draupnir.protectedRoomsSet.setRoomMembership,
        });
    },
});
//# sourceMappingURL=StandardUserConsequencesRenderer.js.map