"use strict";
// Copyright 2022 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirRedactCommand = void 0;
exports.redactEvent = redactEvent;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const utils_1 = require("../utils");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const DraupnirCommandPrerequisites_1 = require("./DraupnirCommandPrerequisites");
async function redactEvent(draupnir, reference, reason) {
    const resolvedRoom = await (0, matrix_protection_suite_for_matrix_bot_sdk_1.resolveRoomReferenceSafe)(draupnir.client, reference.reference);
    if ((0, matrix_protection_suite_1.isError)(resolvedRoom)) {
        return resolvedRoom;
    }
    await draupnir.client.redactEvent(resolvedRoom.ok.toRoomIDOrAlias(), reference.eventID, reason);
    return (0, matrix_protection_suite_1.Ok)(undefined);
}
exports.DraupnirRedactCommand = (0, interface_manager_1.describeCommand)({
    summary: "Redacts either a users's recent messagaes within protected rooms or a specific message shared with the bot.",
    parameters: (0, interface_manager_1.tuple)({
        name: "entity",
        acceptor: {
            schemaType: interface_manager_1.PresentationSchemaType.Union,
            variants: [
                interface_manager_1.MatrixUserIDPresentationType,
                interface_manager_1.MatrixEventReferencePresentationType,
            ],
        },
    }),
    keywords: {
        keywordDescriptions: {
            limit: {
                acceptor: interface_manager_1.StringPresentationType,
                description: "Limit the number of messages to be redacted per room.",
            },
            room: {
                acceptor: interface_manager_1.MatrixRoomReferencePresentationSchema,
                description: "Allows the command to be scoped to just one protected room.",
            },
        },
    },
    rest: {
        name: "reason",
        acceptor: interface_manager_1.StringPresentationType,
    },
    async executor(draupnir, _info, keywords, reasonParts, entity) {
        const reason = reasonParts.join(" ");
        if (entity instanceof matrix_basic_types_1.MatrixEventViaAlias ||
            entity instanceof matrix_basic_types_1.MatrixEventViaRoomID) {
            return await redactEvent(draupnir, entity, reason);
        }
        const rawLimit = keywords.getKeywordValue("limit", undefined);
        const limit = rawLimit === undefined ? undefined : Number.parseInt(rawLimit, 10);
        const restrictToRoomReference = keywords.getKeywordValue("room", undefined);
        const restrictToRoom = restrictToRoomReference
            ? await (0, matrix_protection_suite_for_matrix_bot_sdk_1.resolveRoomReferenceSafe)(draupnir.client, restrictToRoomReference)
            : undefined;
        if (restrictToRoom !== undefined && (0, matrix_protection_suite_1.isError)(restrictToRoom)) {
            return restrictToRoom;
        }
        const roomsToRedactWithin = restrictToRoom === undefined
            ? draupnir.protectedRoomsSet.allProtectedRooms
            : [restrictToRoom.ok];
        await (0, utils_1.redactUserMessagesIn)(draupnir.client, draupnir.managementRoomOutput, entity.toString(), roomsToRedactWithin.map((room) => room.toRoomIDOrAlias()), limit, draupnir.config.noop);
        return (0, matrix_protection_suite_1.Ok)(undefined);
    },
});
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirRedactCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
});
//# sourceMappingURL=RedactCommand.js.map