"use strict";
// Copyright 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirPolicyRemoveCommand = void 0;
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typescript_result_1 = require("@gnuxie/typescript-result");
const Ban_1 = require("../Ban");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
const RecommendationSpec = Object.freeze({
    ban: "ban",
    takedown: "takedown",
});
function recommendationFromSpec(spec) {
    switch (spec) {
        case RecommendationSpec.ban:
            return (0, matrix_protection_suite_1.Ok)(matrix_protection_suite_1.Recommendation.Ban);
        case RecommendationSpec.takedown:
            return (0, matrix_protection_suite_1.Ok)(matrix_protection_suite_1.Recommendation.Takedown);
        default:
            return typescript_result_1.ResultError.Result(`Unknown recommendation provided: ${spec}`);
    }
}
function ruleTypeFromEntity(entity) {
    if (entity instanceof matrix_basic_types_1.MatrixRoomID) {
        return matrix_protection_suite_1.PolicyRuleType.Room;
    }
    else if (entity instanceof matrix_basic_types_1.MatrixUserID) {
        return matrix_protection_suite_1.PolicyRuleType.User;
    }
    else {
        return matrix_protection_suite_1.PolicyRuleType.Server;
    }
}
exports.DraupnirPolicyRemoveCommand = (0, interface_manager_1.describeCommand)({
    summary: "Removes a specific policy from a policy room by using the entity as an exact literal alongside the recommendation to find any relevant policies.\
  This is different to the unban command in that it will not try to find all policies that match the entity.",
    parameters: (0, interface_manager_1.tuple)({
        name: "entity",
        description: "The entity for which the policy exists for",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixRoomIDPresentationType, interface_manager_1.StringPresentationType, interface_manager_1.MatrixUserIDPresentationType),
    }, {
        name: "list",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
        prompt: async function ({ policyRoomManager, clientUserID }) {
            return (0, matrix_protection_suite_1.Ok)({
                suggestions: policyRoomManager
                    .getEditablePolicyRoomIDs(clientUserID, matrix_protection_suite_1.PolicyRuleType.User)
                    .map((room) => interface_manager_1.MatrixRoomIDPresentationType.wrap(room)),
            });
        },
    }, {
        name: "recommendation",
        acceptor: interface_manager_1.StringPresentationType,
        async prompt() {
            return (0, matrix_protection_suite_1.Ok)({
                suggestions: Object.values(RecommendationSpec).map((spec) => interface_manager_1.StringPresentationType.wrap(spec)),
            });
        },
    }),
    keywords: {
        keywordDescriptions: {
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without requiring confirmation or a preview",
            },
        },
    },
    async executor(draupnir, _info, keywords, _rest, entity, policyRoomDesignator, recommendationSpec) {
        const recommendationResult = recommendationFromSpec(recommendationSpec);
        if ((0, typescript_result_1.isError)(recommendationResult)) {
            return recommendationResult;
        }
        const policyRoomReference = typeof policyRoomDesignator === "string"
            ? (0, matrix_protection_suite_1.Ok)(draupnir.protectedRoomsSet.watchedPolicyRooms.findPolicyRoomFromShortcode(policyRoomDesignator)?.room)
            : (0, matrix_protection_suite_1.Ok)(policyRoomDesignator);
        if ((0, typescript_result_1.isError)(policyRoomReference)) {
            return policyRoomReference;
        }
        if (policyRoomReference.ok === undefined) {
            return typescript_result_1.ResultError.Result(`Unable to find a policy room from the shortcode ${policyRoomDesignator.toString()}`);
        }
        const policyRoomEditor = await (0, Ban_1.findPolicyRoomEditorFromRoomReference)(draupnir.clientPlatform.toRoomResolver(), draupnir.policyRoomManager, policyRoomReference.ok);
        if ((0, typescript_result_1.isError)(policyRoomEditor)) {
            return policyRoomEditor.elaborate("Unable to get an editor for the provided policy room reference");
        }
        const policyRoomRevisionIssuer = await draupnir.policyRoomManager.getPolicyRoomRevisionIssuer(policyRoomEditor.ok.room);
        if ((0, typescript_result_1.isError)(policyRoomRevisionIssuer)) {
            return policyRoomRevisionIssuer;
        }
        const policiesToRemove = policyRoomRevisionIssuer.ok.currentRevision
            .allRulesMatchingEntity(entity.toString(), {
            recommendation: recommendationResult.ok,
            type: ruleTypeFromEntity(entity),
            searchHashedRules: true,
        })
            .filter((rule) => rule.matchType === matrix_protection_suite_1.PolicyRuleMatchType.Glob
            ? rule.entity === entity.toString()
            : true);
        const isPreview = !keywords.getKeywordValue("no-confirm", false);
        const removalsBuilder = new matrix_protection_suite_1.StringTypeResultBuilder();
        if (!isPreview) {
            for (const policy of policiesToRemove) {
                const removalResult = await policyRoomEditor.ok.removePolicyByStateKey(ruleTypeFromEntity(entity), policy.sourceEvent.state_key);
                removalsBuilder.addResult(policy.sourceEvent.event_id, removalResult);
            }
        }
        return (0, matrix_protection_suite_1.Ok)({
            policies: policiesToRemove,
            entity,
            policyRoom: policyRoomEditor.ok.room,
            recommendation: recommendationResult.ok,
            policyType: ruleTypeFromEntity(entity),
            isPreview,
            removals: removalsBuilder.getResult(),
        });
    },
});
function renderPolicyRemovalPolicy(policy) {
    // We intentionally include the type from the source event for the situation
    // where policies of legacy types get removed.
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "sender"),
            ": ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, policy.sourceEvent.sender)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "reason"),
            ":",
            " ",
            policy.reason === undefined ? (interface_manager_1.DeadDocumentJSX.JSXFactory("span", null, "no reason was supplied")) : (interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, policy.reason))),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "type"),
            ": ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, policy.sourceEvent.type)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "event_id"),
            ": ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, policy.sourceEvent.event_id)),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "state_key"),
            ": ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, policy.sourceEvent.state_key))));
}
function renderPreview(preview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("root", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("h4", null,
            "The following policies",
            " ",
            preview.isPreview ? "will be removed" : "have been removed",
            " that contain the literal ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.entity.toString()),
            " for the policy type ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.policyType),
            " and recommendation",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.recommendation),
            ":"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, preview.policies.map(renderPolicyRemovalPolicy))));
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirPolicyRemoveCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        return (0, matrix_protection_suite_1.Ok)(renderPreview(commandResult.ok));
    },
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        return (0, matrix_protection_suite_1.Ok)(renderPreview(commandResult.ok));
    },
});
//# sourceMappingURL=PolicyRemove.js.map