"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirUnbanCommand = void 0;
exports.renderUnbanMembersPreview = renderUnbanMembersPreview;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const interface_manager_1 = require("@the-draupnir-project/interface-manager");
const typescript_result_1 = require("@gnuxie/typescript-result");
const DraupnirCommandPrerequisites_1 = require("../DraupnirCommandPrerequisites");
const UnbanUsers_1 = require("./UnbanUsers");
const UnbanEntity_1 = require("./UnbanEntity");
const Rules_1 = require("../Rules");
const mps_interface_adaptor_1 = require("@the-draupnir-project/mps-interface-adaptor");
exports.DraupnirUnbanCommand = (0, interface_manager_1.describeCommand)({
    summary: "Removes an entity from a policy list.",
    parameters: (0, interface_manager_1.tuple)({
        name: "entity",
        description: "The entity to ban. This can be a user ID, room ID, or server name.",
        acceptor: (0, interface_manager_1.union)(interface_manager_1.MatrixUserIDPresentationType, interface_manager_1.MatrixRoomReferencePresentationSchema, interface_manager_1.StringPresentationType),
    }),
    keywords: {
        keywordDescriptions: {
            // This is a legacy option to unban the user from all rooms that we now ignore just so providing the option doesn't
            // cause an error.
            true: {
                isFlag: true,
                description: "Legacy, now redundant option to unban the user from all rooms.",
            },
            invite: {
                isFlag: true,
                description: "Re-invite the unbanned user to any rooms they were unbanned from.",
            },
            "no-confirm": {
                isFlag: true,
                description: "Runs the command without the preview of the unban and the confirmation prompt.",
            },
        },
    },
    async executor({ roomInviter, roomUnbanner, policyRoomManager, watchedPolicyRooms, unlistedUserRedactionQueue, setRoomMembership, }, _info, keywords, _rest, entity) {
        const isNoConfirm = keywords.getKeywordValue("no-confirm", false);
        const inviteMembers = keywords.getKeywordValue("invite", false) ?? false;
        if (entity instanceof matrix_basic_types_1.MatrixUserID) {
            const unbanInformation = (0, UnbanUsers_1.findUnbanInformationForMember)(setRoomMembership, entity, watchedPolicyRooms, { inviteMembers });
            if (!isNoConfirm) {
                return (0, matrix_protection_suite_1.Ok)(unbanInformation);
            }
            else {
                return await (0, UnbanUsers_1.unbanMembers)(unbanInformation, {
                    roomInviter,
                    roomUnbanner,
                    policyRoomManager,
                    unlistedUserRedactionQueue,
                }, { inviteMembers });
            }
        }
        else {
            const unbanPreview = (0, UnbanEntity_1.findPoliciesToRemove)(entity.toString(), watchedPolicyRooms);
            if (!isNoConfirm) {
                return (0, matrix_protection_suite_1.Ok)(unbanPreview);
            }
            else {
                return await (0, UnbanEntity_1.unbanEntity)(entity.toString(), policyRoomManager, unbanPreview);
            }
        }
    },
});
DraupnirCommandPrerequisites_1.DraupnirContextToCommandContextTranslator.registerTranslation(exports.DraupnirUnbanCommand, function (draupnir) {
    return {
        policyRoomManager: draupnir.policyRoomManager,
        watchedPolicyRooms: draupnir.protectedRoomsSet.watchedPolicyRooms,
        roomResolver: draupnir.clientPlatform.toRoomResolver(),
        clientUserID: draupnir.clientUserID,
        setRoomMembership: draupnir.protectedRoomsSet.setRoomMembership,
        setMembership: draupnir.protectedRoomsSet.setMembership,
        setPoliciesMatchingMembership: draupnir.protectedRoomsSet.setPoliciesMatchingMembership
            .currentRevision,
        managementRoomOutput: draupnir.managementRoomOutput,
        noop: draupnir.config.noop,
        roomUnbanner: draupnir.clientPlatform.toRoomUnbanner(),
        unlistedUserRedactionQueue: draupnir.unlistedUserRedactionQueue,
        roomInviter: draupnir.clientPlatform.toRoomInviter(),
    };
});
function policyRemovalText() {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        "If you wish to remove only a specific policy, please use the",
        " ",
        interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, "policy remove"),
        " command."));
}
function renderPoliciesToRemove(policyMatches) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        "The following policies will be removed:",
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, policyMatches.map((list) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, (0, Rules_1.renderListRules)(list)))))));
}
function renderUnbanEntityPreview(preview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        "You are about to unban the entity ",
        preview.entity.toString(),
        ", do you want to continue?",
        renderPoliciesToRemove(preview.policyMatchesToRemove)));
}
function renderMemberRoomsUnbanPreview(memberRooms) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            (0, mps_interface_adaptor_1.renderMentionPill)(memberRooms.member, memberRooms.member),
            " will be unbanned from ",
            memberRooms.roomsBannedFrom.length,
            " rooms"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, memberRooms.roomsBannedFrom.map((room) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, (0, mps_interface_adaptor_1.renderRoomPill)(room)))))));
}
function renderMemberRoomsInvitePreview(memberRooms) {
    if (memberRooms.roomsToInviteTo.length === 0) {
        return interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null);
    }
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null,
            (0, mps_interface_adaptor_1.renderMentionPill)(memberRooms.member, memberRooms.member),
            " will be invited back to ",
            memberRooms.roomsToInviteTo.length,
            " rooms"),
        interface_manager_1.DeadDocumentJSX.JSXFactory("ul", null, memberRooms.roomsToInviteTo.map((room) => (interface_manager_1.DeadDocumentJSX.JSXFactory("li", null, (0, mps_interface_adaptor_1.renderRoomPill)(room)))))));
}
function renderMemberRoomsPreview(memberRooms) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        renderMemberRoomsUnbanPreview(memberRooms),
        renderMemberRoomsInvitePreview(memberRooms)));
}
function renderUnbanMembersPreview(preview) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        preview.entity.isContainingGlobCharacters() ? (interface_manager_1.DeadDocumentJSX.JSXFactory("h4", null,
            "You are about to unban users matching the glob",
            " ",
            interface_manager_1.DeadDocumentJSX.JSXFactory("code", null, preview.entity.toString()))) : (interface_manager_1.DeadDocumentJSX.JSXFactory("h4", null,
            "You are about to unban",
            " ",
            (0, mps_interface_adaptor_1.renderMentionPill)(preview.entity.toString(), preview.entity.toString()))),
        renderPoliciesToRemove(preview.policyMatchesToRemove),
        preview.membersToUnban.length,
        " users will be unbanned:",
        preview.membersToUnban.map(renderMemberRoomsPreview),
        interface_manager_1.DeadDocumentJSX.JSXFactory("br", null),
        policyRemovalText()));
}
function renderPolicyRemovalResult(result) {
    if (result.policyRemovalResult.map.size === 0) {
        return interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null);
    }
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null, (0, mps_interface_adaptor_1.renderRoomSetResult)(result.policyRemovalResult, {
        summary: interface_manager_1.DeadDocumentJSX.JSXFactory("span", null, "Policies were removed from the following rooms:"),
    })));
}
function renderUnbanEntityResult(result) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        interface_manager_1.DeadDocumentJSX.JSXFactory("details", null,
            interface_manager_1.DeadDocumentJSX.JSXFactory("summary", null, "The following policies were found banning this entity"),
            renderPolicyRemovalResult(result)),
        policyRemovalText()));
}
function renderUnbanMembersResult(result) {
    return (interface_manager_1.DeadDocumentJSX.JSXFactory("fragment", null,
        renderPolicyRemovalResult(result),
        [...result.usersUnbanned.map.entries()].map(([userID, roomSetResult]) => (0, mps_interface_adaptor_1.renderRoomSetResult)(roomSetResult, {
            summary: (interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                (0, mps_interface_adaptor_1.renderMentionPill)(userID, userID),
                " was unbanned from",
                " ",
                roomSetResult.map.size,
                " rooms:")),
        })),
        [...result.usersInvited.map.entries()].map(([userID, roomSetResult]) => (0, mps_interface_adaptor_1.renderRoomSetResult)(roomSetResult, {
            summary: (interface_manager_1.DeadDocumentJSX.JSXFactory("span", null,
                (0, mps_interface_adaptor_1.renderMentionPill)(userID, userID),
                " was invited back to",
                " ",
                roomSetResult.map.size,
                " rooms:")),
        }))));
}
DraupnirCommandPrerequisites_1.DraupnirInterfaceAdaptor.describeRenderer(exports.DraupnirUnbanCommand, {
    isAlwaysSupposedToUseDefaultRenderer: true,
    confirmationPromptJSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        else if ("membersToUnban" in commandResult.ok) {
            return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderUnbanMembersPreview(commandResult.ok)));
        }
        else {
            return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderUnbanEntityPreview(commandResult.ok)));
        }
    },
    JSXRenderer(commandResult) {
        if ((0, typescript_result_1.isError)(commandResult)) {
            return (0, matrix_protection_suite_1.Ok)(undefined);
        }
        else if ("usersUnbanned" in commandResult.ok) {
            return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderUnbanMembersResult(commandResult.ok)));
        }
        else if ("policyRemovalResult" in commandResult.ok) {
            return (0, matrix_protection_suite_1.Ok)(interface_manager_1.DeadDocumentJSX.JSXFactory("root", null, renderUnbanEntityResult(commandResult.ok)));
        }
        else {
            throw new TypeError("The unban command is quite broken you should tell the developers");
        }
    },
});
//# sourceMappingURL=Unban.js.map