"use strict";
// SPDX-FileCopyrightText: 2023 - 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.DraupnirFactory = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const Draupnir_1 = require("../Draupnir");
const matrix_protection_suite_for_matrix_bot_sdk_1 = require("matrix-protection-suite-for-matrix-bot-sdk");
const DraupnirProtectedRoomsSet_1 = require("./DraupnirProtectedRoomsSet");
const DraupnirSafeMode_1 = require("../safemode/DraupnirSafeMode");
const ManagementRoomDetail_1 = require("../managementroom/ManagementRoomDetail");
const DraupnirStores_1 = require("../backingstore/DraupnirStores");
const SynapseHttpAntispam_1 = require("../webapis/SynapseHTTPAntispam/SynapseHttpAntispam");
const log = new matrix_protection_suite_1.Logger("DraupnirFactory");
async function safelyFetchProfile(client, clientUserID) {
    const clientProfileResult = await client.getUserProfile(clientUserID).then((value) => (0, matrix_protection_suite_1.Ok)(value), (error) => (0, matrix_protection_suite_for_matrix_bot_sdk_1.resultifyBotSDKRequestErrorWith404AsUndefined)(error));
    // We opt to report to the log rather than fail outright. Because this is on the critical startup path
    // and when we did crash, we had unexplained server behaviour https://github.com/the-draupnir-project/Draupnir/issues/703
    if ((0, matrix_protection_suite_1.isError)(clientProfileResult)) {
        log.error("Unable to fetch Draupnir's profile information, please report this error to the developers", clientProfileResult.error);
        return undefined;
    }
    else {
        return clientProfileResult.ok;
    }
}
class DraupnirFactory {
    constructor(clientsInRoomMap, clientCapabilityFactory, clientProvider, roomStateManagerFactory, stores) {
        this.clientsInRoomMap = clientsInRoomMap;
        this.clientCapabilityFactory = clientCapabilityFactory;
        this.clientProvider = clientProvider;
        this.roomStateManagerFactory = roomStateManagerFactory;
        this.stores = stores;
        // nothing to do.
    }
    async makeDraupnir(clientUserID, managementRoom, config, toggle) {
        const client = await this.clientProvider(clientUserID);
        const clientRooms = await this.clientsInRoomMap.makeClientRooms(clientUserID, async () => (0, matrix_protection_suite_for_matrix_bot_sdk_1.joinedRoomsSafe)(client));
        if ((0, matrix_protection_suite_1.isError)(clientRooms)) {
            return clientRooms;
        }
        const roomStateManager = await this.roomStateManagerFactory.getRoomStateManager(clientUserID);
        const policyRoomManager = await this.roomStateManagerFactory.getPolicyRoomManager(clientUserID);
        const roomMembershipManager = await this.roomStateManagerFactory.getRoomMembershipManager(clientUserID);
        const clientPlatform = this.clientCapabilityFactory.makeClientPlatform(clientUserID, client);
        const configLogTracker = new matrix_protection_suite_1.StandardLoggableConfigTracker();
        const protectedRoomsSet = await (0, DraupnirProtectedRoomsSet_1.makeProtectedRoomsSet)(managementRoom, roomStateManager, policyRoomManager, roomMembershipManager, client, clientPlatform, clientUserID, config, configLogTracker);
        if ((0, matrix_protection_suite_1.isError)(protectedRoomsSet)) {
            return protectedRoomsSet;
        }
        const managementRoomMembership = await this.roomStateManagerFactory.getRoomMembershipRevisionIssuer(managementRoom, clientUserID);
        if ((0, matrix_protection_suite_1.isError)(managementRoomMembership)) {
            return managementRoomMembership.elaborate("Failed to get room membership revision issuer for the management room");
        }
        const managementRoomState = await this.roomStateManagerFactory.getRoomStateRevisionIssuer(managementRoom, clientUserID);
        if ((0, matrix_protection_suite_1.isError)(managementRoomState)) {
            return managementRoomState.elaborate("Failed to get room state revision issuer for the management room");
        }
        const managementRoomDetail = new ManagementRoomDetail_1.StandardManagementRoomDetail(managementRoom, managementRoomMembership.ok, managementRoomState.ok);
        // synapesHTTPAntispam is made here so that the listeners get destroyed
        // when draupnir is destroyed.
        const synapseHTTPAntispam = config.web.synapseHTTPAntispam.enabled
            ? new SynapseHttpAntispam_1.SynapseHttpAntispam(config.web.synapseHTTPAntispam.authorization)
            : undefined;
        const clientProfile = await safelyFetchProfile(client, clientUserID);
        return await Draupnir_1.Draupnir.makeDraupnirBot(client, clientUserID, clientProfile?.displayname ?? clientUserID, clientPlatform, managementRoomDetail, clientRooms.ok, protectedRoomsSet.ok, roomStateManager, policyRoomManager, roomMembershipManager, config, configLogTracker, toggle, (0, DraupnirStores_1.createDraupnirStores)(this.stores), 
        // synapseHTTPAntispam is only available in bot mode.
        synapseHTTPAntispam);
    }
    async makeSafeModeDraupnir(clientUserID, managementRoom, config, cause, toggle) {
        const client = await this.clientProvider(clientUserID);
        const clientRooms = await this.clientsInRoomMap.makeClientRooms(clientUserID, async () => (0, matrix_protection_suite_for_matrix_bot_sdk_1.joinedRoomsSafe)(client));
        if ((0, matrix_protection_suite_1.isError)(clientRooms)) {
            return clientRooms;
        }
        const clientPlatform = this.clientCapabilityFactory.makeClientPlatform(clientUserID, client);
        const clientProfile = await safelyFetchProfile(client, clientUserID);
        return (0, matrix_protection_suite_1.Ok)(new DraupnirSafeMode_1.SafeModeDraupnir(cause, client, clientUserID, clientProfile?.displayname ?? clientUserID, clientPlatform, managementRoom, clientRooms.ok, toggle, config));
    }
}
exports.DraupnirFactory = DraupnirFactory;
//# sourceMappingURL=DraupnirFactory.js.map