"use strict";
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeywordParser = void 0;
exports.describeKeywordParameters = describeKeywordParameters;
const typescript_result_1 = require("@gnuxie/typescript-result");
const TextPresentationRenderer_1 = require("../TextReader/TextPresentationRenderer");
const Keyword_1 = require("./Keyword");
const ParseErrors_1 = require("./ParseErrors");
const ParsedKeywords_1 = require("./ParsedKeywords");
const PresentationSchema_1 = require("./PresentationSchema");
/**
 * A helper that gets instantiated for each command invoccation to parse and build
 * the map representing the association between keywords and their properties.
 */
class KeywordParser {
    constructor(description) {
        this.description = description;
        this.arguments = new Map();
    }
    getKeywords() {
        return new ParsedKeywords_1.StandardParsedKeywords(this.description, this.arguments);
    }
    readKeywordAssociatedProperty(keyword, partialCommand) {
        const stream = partialCommand.stream;
        const nextItem = stream.peekItem();
        if (nextItem !== undefined && !(nextItem.object instanceof Keyword_1.Keyword)) {
            const acceptedPresentation = (0, PresentationSchema_1.acceptPresentation)(keyword.acceptor, partialCommand.commandTable, nextItem);
            if (acceptedPresentation !== undefined) {
                stream.readItem(); // consume the presentation from the stream.
                return (0, typescript_result_1.Ok)(acceptedPresentation);
            }
            else {
                return ParseErrors_1.ArgumentParseError.Result(`Was expecting a match for the presentation type: ${(0, PresentationSchema_1.printPresentationSchema)(keyword.acceptor)} but got ${TextPresentationRenderer_1.TextPresentationRenderer.render(nextItem)}.`, {
                    parameter: keyword,
                    partialCommand,
                });
            }
        }
        else {
            if (!keyword.isFlag) {
                return ParseErrors_1.ArgumentParseError.Result(`An associated argument was not provided for the keyword ${keyword.name}.`, { parameter: keyword, partialCommand });
            }
            else {
                return (0, typescript_result_1.Ok)(true);
            }
        }
    }
    parseKeywords(partialCommand) {
        var _a;
        const stream = partialCommand.stream;
        while (((_a = stream.peekItem()) === null || _a === void 0 ? void 0 : _a.object) instanceof Keyword_1.Keyword) {
            const item = stream.readItem();
            const description = this.description.keywordDescriptions[item.object.designator];
            if (description === undefined) {
                if (this.description.allowOtherKeys) {
                    throw new TypeError("Allow other keys is umimplemented");
                    // i don't think this can be implemented,
                    // how do you tell an extra key is a flag or has an associated
                    // property?
                }
                else {
                    return ParseErrors_1.UnexpectedArgumentError.Result(`Encountered unexpected keyword argument: ${item.object.designator}`, { partialCommand });
                }
            }
            else {
                const associatedPropertyResult = this.readKeywordAssociatedProperty(
                // idk why typescript is bottoming out here but whatever.
                description, partialCommand);
                if ((0, typescript_result_1.isError)(associatedPropertyResult)) {
                    return associatedPropertyResult;
                }
                else {
                    this.arguments.set(description.name, associatedPropertyResult.ok);
                }
            }
        }
        return (0, typescript_result_1.Ok)(this);
    }
    parseRest(partialCommand, restDescription) {
        const stream = partialCommand.stream;
        if (restDescription !== undefined) {
            return restDescription.parseRest(partialCommand, this);
        }
        else {
            const result = this.parseKeywords(partialCommand);
            if ((0, typescript_result_1.isError)(result)) {
                return result;
            }
            if (stream.peekItem() !== undefined) {
                const textRendering = (() => {
                    var _a;
                    try {
                        return TextPresentationRenderer_1.TextPresentationRenderer.render(stream.peekItem());
                    }
                    catch (_e) {
                        return `${JSON.stringify((_a = stream.peekItem()) === null || _a === void 0 ? void 0 : _a.object)} (fallback representation)`;
                    }
                })();
                return ParseErrors_1.UnexpectedArgumentError.Result(`There is an unexpected non-keyword argument: ${textRendering}`, { partialCommand });
            }
            else {
                return (0, typescript_result_1.Ok)(undefined);
            }
        }
    }
}
exports.KeywordParser = KeywordParser;
function describeKeywordProperty(name, property) {
    var _a;
    if (property.acceptor === undefined) {
        if (!property.isFlag) {
            throw new TypeError("An acceptor is required if the property is not a flag.");
        }
    }
    const acceptor = ((acceptor) => {
        if (acceptor === undefined) {
            return PresentationSchema_1.TopPresentationSchema;
        }
        else if ("schemaType" in acceptor) {
            return acceptor;
        }
        else {
            return {
                schemaType: PresentationSchema_1.PresentationSchemaType.Single,
                presentationType: acceptor,
            };
        }
    })(property.acceptor);
    return {
        name,
        isFlag: (_a = property.isFlag) !== null && _a !== void 0 ? _a : false,
        acceptor,
        description: property.description,
    };
}
function describeKeywordParameters(options) {
    var _a;
    const keywordDescriptions = {};
    for (const [name, property] of Object.entries(options.keywordDescriptions)) {
        keywordDescriptions[name] = describeKeywordProperty(name, property);
    }
    return {
        keywordDescriptions: keywordDescriptions,
        allowOtherKeys: (_a = options.allowOtherKeys) !== null && _a !== void 0 ? _a : false,
        getParser() {
            return new KeywordParser(this);
        },
    };
}
//# sourceMappingURL=KeywordParameterDescription.js.map