"use strict";
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const describeCommand_1 = require("./describeCommand");
const typescript_result_1 = require("@gnuxie/typescript-result");
const TextReader_1 = require("../TextReader");
const ParsedKeywords_1 = require("./ParsedKeywords");
const ParameterParsing_1 = require("./ParameterParsing");
const Command_1 = require("./Command");
const PresentationStream_1 = require("./PresentationStream");
const CommandTable_1 = require("./CommandTable");
const PresentationSchema_1 = require("./PresentationSchema");
it("Can define and execute commands.", async function () {
    // i think what we have to do is split describeCommand into two parts :(
    // The first extracts the parameters and possibly accepts the `context` as a type parmater.
    // then the function it returns accepts the executor.
    const BanCommand = (0, describeCommand_1.describeCommand)({
        summary: "Ban a user from a room",
        async executor(context, _info, _keywords, _rest, user, room) {
            return await context.banUser(room, user);
        },
        parameters: (0, ParameterParsing_1.tuple)({
            name: "user",
            acceptor: TextReader_1.MatrixUserIDPresentationType,
        }, {
            name: "target room",
            acceptor: TextReader_1.MatrixRoomReferencePresentationSchema,
            async prompt(context) {
                return (0, typescript_result_1.Ok)({
                    suggestions: context
                        .getProtectedRooms()
                        .map((room) => TextReader_1.MatrixRoomIDPresentationType.wrap(room)),
                });
            },
        }),
    });
    const banResult = await BanCommand.executor({
        async banUser(room, user) {
            expect(room.toRoomIDOrAlias()).toBe("!foo:example.com");
            expect(user.toString()).toBe("@foo:example.com");
            return (0, typescript_result_1.Ok)(true);
        },
        getProtectedRooms() {
            return [new matrix_basic_types_1.MatrixRoomID("!foo:example.com")];
        },
    }, {}, new ParsedKeywords_1.StandardParsedKeywords(BanCommand.parametersDescription.keywords, new Map()), [], new matrix_basic_types_1.MatrixUserID("@foo:example.com"), matrix_basic_types_1.MatrixRoomReference.fromRoomID("!foo:example.com", []));
    expect((0, typescript_result_1.isOk)(banResult)).toBe(true);
});
it("Can define keyword arguments.", async function () {
    var _a;
    const KeywordsCommandTest = (0, describeCommand_1.describeCommand)({
        summary: "A command to test keyword arguments",
        async executor(_context, _info, _keywords) {
            return (0, typescript_result_1.Ok)(undefined);
        },
        parameters: [],
        keywords: {
            keywordDescriptions: {
                "dry-run": {
                    isFlag: true,
                    description: "Runs the kick command without actually removing any users.",
                },
                glob: {
                    isFlag: true,
                    description: "Allows globs to be used to kick several users from rooms.",
                },
                room: {
                    acceptor: TextReader_1.MatrixRoomReferencePresentationSchema,
                    description: "Allows the command to be scoped to just one protected room.",
                },
            },
        },
    });
    const parseResults = KeywordsCommandTest.parametersDescription.parse((0, Command_1.makePartialCommand)(new PresentationStream_1.StandardPresentationArgumentStream((0, TextReader_1.readCommand)(`--dry-run --room !foo:example.com`)), KeywordsCommandTest, new CommandTable_1.StandardCommandTable(Symbol("KeywordsCommandTest")), []));
    if ((0, typescript_result_1.isError)(parseResults)) {
        throw new TypeError(`Failed to parse for some reason: ${parseResults.error.mostRelevantElaboration}`);
    }
    const keywords = parseResults.ok.keywords;
    expect(keywords.getKeywordValue("dry-run")).toBe(true);
    expect(keywords.getKeywordValue("glob", false)).toBe(false);
    expect((_a = keywords.getKeywordValue("room")) === null || _a === void 0 ? void 0 : _a.toRoomIDOrAlias()).toEqual("!foo:example.com");
});
it("end to end test a command that parses mxids", async function () {
    const tableName = Symbol("ParseTest");
    const testTable = new CommandTable_1.StandardCommandTable(tableName);
    const helpCommand = (0, describeCommand_1.describeCommand)({
        summary: "Mimicks the help command",
        parameters: [],
        async executor() {
            return (0, typescript_result_1.Ok)("here is your help");
        },
    });
    const unbanCommand = (0, describeCommand_1.describeCommand)({
        summary: "Mimicks the unban command",
        parameters: (0, ParameterParsing_1.tuple)({
            name: "entity",
            acceptor: (0, PresentationSchema_1.union)(TextReader_1.MatrixUserIDPresentationType, TextReader_1.MatrixRoomReferencePresentationSchema, TextReader_1.StringPresentationType),
        }),
        async executor(_context, _info, _keywords, _rest, entity) {
            return (0, typescript_result_1.Ok)(entity);
        },
    });
    testTable.internCommand(unbanCommand, [
        "draupnir",
        "unban",
    ]);
    const dispatcher = new TextReader_1.StandardJSInterfaceCommandDispatcher(testTable, helpCommand, undefined, { commandNormaliser: (body) => body });
    const result = await dispatcher.invokeCommandFromBody({ commandSender: "@test:localhost" }, "draupnir unban @spam:example.com");
    if ((0, typescript_result_1.isError)(result)) {
        throw new TypeError(`Not supposed to be error mate`);
    }
    expect(result.ok).toBeInstanceOf(matrix_basic_types_1.MatrixUserID);
});
//# sourceMappingURL=describeCommand.test.js.map