"use strict";
// Copyright 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from @the-draupnir-project/interface-manager
// https://github.com/the-draupnir-project/interface-manager
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MatrixEventReferencePresentationType = exports.MatrixUserIDPresentationType = exports.MatrixRoomReferencePresentationSchema = exports.MatrixRoomAliasPresentationType = exports.MatrixRoomIDPresentationType = exports.KeywordPresentationType = exports.BooleanPresentationType = exports.NumberPresentationType = exports.StringPresentationType = void 0;
exports.makeMatrixEventReferencePresentation = makeMatrixEventReferencePresentation;
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
const Presentation_1 = require("../Command/Presentation");
const Keyword_1 = require("../Command/Keyword");
const TextPresentationRenderer_1 = require("./TextPresentationRenderer");
const PresentationSchema_1 = require("../Command/PresentationSchema");
/**
 * If you are wondering why commands specify on presentation type and not
 * their actual type, then imagine that you have a a presentation type
 * for a person's name, and you render that in handwriting.
 * The person's name is really just a string, and it'd be wrong to be able to
 * get any string with that command as a name. OTOH, I don't really see the
 * point in that? WHy not just make a real type for a person's name?
 */
exports.StringPresentationType = (0, Presentation_1.definePresentationType)({
    name: "string",
    validator: function (value) {
        return typeof value === "string";
    },
    wrap(string) {
        return Object.freeze({
            object: string,
            presentationType: exports.StringPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.StringPresentationType, function (presentation) {
    return presentation.object;
});
exports.NumberPresentationType = (0, Presentation_1.definePresentationType)({
    name: "number",
    validator: function (value) {
        return typeof value === "number";
    },
    wrap(number) {
        return Object.freeze({
            object: number,
            presentationType: exports.NumberPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.NumberPresentationType, function (presentation) {
    return presentation.object.toString();
});
exports.BooleanPresentationType = (0, Presentation_1.definePresentationType)({
    name: "boolean",
    validator: function (value) {
        return typeof value === "boolean";
    },
    wrap(boolean) {
        return Object.freeze({
            object: boolean,
            presentationType: exports.BooleanPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.BooleanPresentationType, function (presentation) {
    return presentation.object ? "true" : "false";
});
exports.KeywordPresentationType = (0, Presentation_1.definePresentationType)({
    name: "Keyword",
    validator: function (value) {
        return value instanceof Keyword_1.Keyword;
    },
    wrap(keyword) {
        return Object.freeze({
            object: keyword,
            presentationType: exports.KeywordPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.KeywordPresentationType, function (presetnation) {
    return `--${presetnation.object.designator}`;
});
exports.MatrixRoomIDPresentationType = (0, Presentation_1.definePresentationType)({
    name: "MatrixRoomID",
    validator: function (value) {
        return value instanceof matrix_basic_types_1.MatrixRoomID;
    },
    wrap(roomID) {
        return Object.freeze({
            object: roomID,
            presentationType: exports.MatrixRoomIDPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.MatrixRoomIDPresentationType, function (presentation) {
    return presentation.object.toRoomIDOrAlias();
});
exports.MatrixRoomAliasPresentationType = (0, Presentation_1.definePresentationType)({
    name: "MatrixRoomAlias",
    validator: function (value) {
        return value instanceof matrix_basic_types_1.MatrixRoomAlias;
    },
    wrap(alias) {
        return Object.freeze({
            object: alias,
            presentationType: exports.MatrixRoomAliasPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.MatrixRoomAliasPresentationType, function (presentation) {
    return presentation.object.toRoomIDOrAlias();
});
exports.MatrixRoomReferencePresentationSchema = (0, PresentationSchema_1.union)(exports.MatrixRoomIDPresentationType, exports.MatrixRoomAliasPresentationType);
exports.MatrixUserIDPresentationType = (0, Presentation_1.definePresentationType)({
    name: "MatrixUserID",
    validator: function (value) {
        return value instanceof matrix_basic_types_1.MatrixUserID;
    },
    wrap(userID) {
        return Object.freeze({
            object: userID,
            presentationType: exports.MatrixUserIDPresentationType,
        });
    },
});
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.MatrixUserIDPresentationType, function (presentation) {
    return presentation.object.toString();
});
exports.MatrixEventReferencePresentationType = (0, Presentation_1.definePresentationType)({
    name: "MatrixEventReference",
    validator: function (value) {
        return (value instanceof matrix_basic_types_1.MatrixEventViaAlias ||
            value instanceof matrix_basic_types_1.MatrixEventViaRoomID);
    },
    wrap(event) {
        return Object.freeze({
            object: event,
            presentationType: exports.MatrixEventReferencePresentationType,
        });
    },
});
function makeMatrixEventReferencePresentation(value) {
    return Object.freeze({
        object: value,
        presentationType: exports.MatrixEventReferencePresentationType,
    });
}
TextPresentationRenderer_1.TextPresentationRenderer.registerPresentationRenderer(exports.MatrixEventReferencePresentationType, function (presentation) {
    return `${presentation.object.reference.toPermalink()}/${presentation.object.eventID}`;
});
//# sourceMappingURL=TextPresentationTypes.js.map