"use strict";
/**
 * Copyright (C) 2022-2023 Gnuxie <Gnuxie@protonmail.com>
 * All rights reserved.
 *
 * This file is modified and is NOT licensed under the Apache License.
 * This modified file incorperates work from mjolnir
 * https://github.com/matrix-org/mjolnir
 * which included the following license notice:

Copyright 2019-2021 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 *
 * However, this file is modified and the modifications in this file
 * are NOT distributed, contributed, committed, or licensed under the Apache License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.BotSDKPolicyRoomManager = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
class BotSDKPolicyRoomManager {
    constructor(clientUserID, client, clientPlatform, factory, joinPreempter) {
        this.clientUserID = clientUserID;
        this.client = client;
        this.clientPlatform = clientPlatform;
        this.factory = factory;
        this.joinPreempter = joinPreempter;
        this.issuedEditors = new matrix_protection_suite_1.InternedInstanceFactory(async (_roomID, room) => {
            const policyRoomRevisionIssuer = await this.factory.getPolicyRoomRevisionIssuer(room, this.clientUserID);
            if ((0, matrix_protection_suite_1.isError)(policyRoomRevisionIssuer)) {
                return policyRoomRevisionIssuer;
            }
            const roomStateRevisionIssuer = await this.factory.getRoomStateRevisionIssuer(room, this.clientUserID);
            if ((0, matrix_protection_suite_1.isError)(roomStateRevisionIssuer)) {
                return roomStateRevisionIssuer;
            }
            const editor = new matrix_protection_suite_1.StandardPolicyRoomEditor(room, policyRoomRevisionIssuer.ok, roomStateRevisionIssuer.ok, this.clientPlatform.toRoomStateEventSender());
            return (0, matrix_protection_suite_1.Ok)(editor);
        });
        // nothing to do.
    }
    async getPolicyRoomEditor(room) {
        return await this.issuedEditors.getInstance(room.toRoomIDOrAlias(), room);
    }
    async createPolicyRoom(shortcode, invite, createRoomOptions) {
        var _a;
        const creator = await this.client.getUserId().then((user) => (0, matrix_protection_suite_1.Ok)((0, matrix_basic_types_1.StringUserID)(user)), (exception) => matrix_protection_suite_1.ActionException.Result('Could not create a list because we could not find the mxid of the list creator.', { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
        if ((0, matrix_protection_suite_1.isError)(creator)) {
            return creator;
        }
        const clientCapabilities = await this.clientPlatform
            .toClientCapabilitiesNegotiation()
            .getClientCapabilities();
        if ((0, matrix_protection_suite_1.isError)(clientCapabilities)) {
            return clientCapabilities.elaborate('Unable to get client capabilities for the policy room creator.');
        }
        const isRoomVersionWithPrivilidgedCreators = matrix_protection_suite_1.RoomVersionMirror.isVersionWithPrivilidgedCreators(clientCapabilities.ok.capabilities['m.room_versions'].default);
        const powerLevels = {
            ban: 50,
            events: {
                'm.room.name': 100,
                'm.room.power_levels': 100,
            },
            events_default: 50, // non-default
            invite: 0,
            kick: 50,
            notifications: {
                room: 20,
            },
            redact: 50,
            state_default: 50,
            users: {
                ...(isRoomVersionWithPrivilidgedCreators ? {} : { [creator.ok]: 100 }),
                ...invite.reduce((users, mxid) => ({ ...users, [mxid]: 50 }), {}),
            },
            users_default: 0,
        };
        const finalRoomCreateOptions = {
            // Support for MSC3784.
            creation_content: {
                type: matrix_protection_suite_1.POLICY_ROOM_TYPE,
            },
            preset: 'public_chat',
            invite,
            initial_state: [
                {
                    type: matrix_protection_suite_1.MJOLNIR_SHORTCODE_EVENT_TYPE,
                    state_key: '',
                    content: { shortcode: shortcode },
                },
            ],
            power_level_content_override: powerLevels,
            ...createRoomOptions,
        };
        // Guard room type in case someone overwrites it when declaring custom creation_content in future code.
        const roomType = (_a = finalRoomCreateOptions.creation_content) === null || _a === void 0 ? void 0 : _a.type;
        if (typeof roomType !== 'string' ||
            !matrix_protection_suite_1.POLICY_ROOM_TYPE_VARIANTS.includes(roomType)) {
            throw new TypeError(`Creating a policy room with a type other than the policy room type is not supported, you probably don't want to do this.`);
        }
        return await this.client.createRoom(finalRoomCreateOptions).then((roomId) => {
            const room = new matrix_basic_types_1.MatrixRoomID(roomId, [(0, matrix_basic_types_1.userServerName)(creator.ok)]);
            this.joinPreempter.preemptTimelineJoin(this.clientUserID, room.toRoomIDOrAlias());
            return (0, matrix_protection_suite_1.Ok)(room);
        }, (exception) => matrix_protection_suite_1.ActionException.Result('Could not create a matrix room to serve as the new policy list.', { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
    }
    getPolicyRuleEvents(room) {
        return this.client.getRoomState(room.toRoomIDOrAlias()).then((events) => (0, matrix_protection_suite_1.Ok)(events.filter(matrix_protection_suite_1.isPolicyRuleEvent)), (exception) => matrix_protection_suite_1.ActionError.Result(`Could not fetch the room state for the policy list ${room.toPermalink()} and so we are unable to fetch any policy rules.`, { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
    }
    async getPolicyRoomRevisionIssuer(room) {
        return await this.factory.getPolicyRoomRevisionIssuer(room, this.clientUserID);
    }
    getEditablePolicyRoomIDs(editor, ruleType) {
        return this.factory.getEditablePolicyRoomIDs(editor, ruleType);
    }
}
exports.BotSDKPolicyRoomManager = BotSDKPolicyRoomManager;
//# sourceMappingURL=PolicyListManager.js.map