"use strict";
/**
 * Copyright (C) 2022-2023 Gnuxie <Gnuxie@protonmail.com>
 * All rights reserved.
 *
 * This file is modified and is NOT licensed under the Apache License.
 * This modified file incorperates work from mjolnir
 * https://github.com/matrix-org/mjolnir
 * which included the following license notice:

Copyright 2019-2021 The Matrix.org Foundation C.I.C.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 *
 * However, this file is modified and the modifications in this file
 * are NOT distributed, contributed, committed, or licensed under the Apache License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynapseAdminClient = exports.AccountRestriction = void 0;
const matrix_protection_suite_1 = require("matrix-protection-suite");
const typebox_1 = require("@sinclair/typebox");
const BotSDKBaseClient_1 = require("../Client/BotSDKBaseClient");
const BlockStatusEndpoint_1 = require("./BlockStatusEndpoint");
const RoomDetailsEndpoint_1 = require("./RoomDetailsEndpoint");
const UserDetailsEndpoint_1 = require("./UserDetailsEndpoint");
const UserRedactionEndpoint_1 = require("./UserRedactionEndpoint");
const typescript_result_1 = require("@gnuxie/typescript-result");
const RoomListEndpoint_1 = require("./RoomListEndpoint");
const ReportPollResponse = typebox_1.Type.Object({
    event_reports: typebox_1.Type.Array(matrix_protection_suite_1.SynapseReport),
    next_token: typebox_1.Type.Optional(typebox_1.Type.Union([typebox_1.Type.Integer(), typebox_1.Type.Null()])),
    total: typebox_1.Type.Optional(typebox_1.Type.Union([typebox_1.Type.Integer(), typebox_1.Type.Null()])),
});
/**
 * An account restriction at the minimum stops the user from sending
 * messages.
 */
var AccountRestriction;
(function (AccountRestriction) {
    AccountRestriction["Suspended"] = "suspended";
    AccountRestriction["Deactivated"] = "deactivated";
    AccountRestriction["ShadowBanned"] = "shadow_banned";
})(AccountRestriction || (exports.AccountRestriction = AccountRestriction = {}));
class SynapseAdminClient {
    constructor(client, clientUserID) {
        this.client = client;
        this.clientUserID = clientUserID;
        // nothing to do.
    }
    async isSynapseAdmin() {
        var _a;
        const endpoint = `/_synapse/admin/v1/users/${encodeURIComponent(this.clientUserID)}/admin`;
        const response = await this.client.doRequest('GET', endpoint).then((value) => (0, matrix_protection_suite_1.Ok)(value), (exception) => matrix_protection_suite_1.ActionException.Result(`Unable to query whether the user ${this.clientUserID} is a Synapse Admin`, { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
        if ((0, matrix_protection_suite_1.isError)(response)) {
            return response;
        }
        const decodedResult = matrix_protection_suite_1.Value.Decode(matrix_protection_suite_1.SynapseAdminGetUserAdminResponse, response.ok);
        if ((0, matrix_protection_suite_1.isError)(decodedResult)) {
            return decodedResult;
        }
        else {
            return (0, matrix_protection_suite_1.Ok)((_a = decodedResult.ok.admin) !== null && _a !== void 0 ? _a : false);
        }
    }
    async deactivateUser(targetUserID, { erase = false } = {}) {
        const endpoint = `/_synapse/admin/v1/deactivate/${encodeURIComponent(targetUserID)}`;
        return await this.client
            .doRequest('POST', endpoint, undefined, { erase })
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), (exception) => matrix_protection_suite_1.ActionException.Result(`Unable to deactivate the user ${targetUserID}`, { exception, exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown }));
    }
    async deleteRoom(roomID, { block = true, ...otherOptions } = {}) {
        const endpoint = `/_synapse/admin/v1/rooms/${encodeURIComponent(roomID)}`;
        return await this.client
            .doRequest('DELETE', endpoint, null, {
            new_room_user_id: this.clientUserID,
            block,
            ...otherOptions,
        })
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), (exception) => matrix_protection_suite_1.ActionException.Result(`Unable to delete the room ${roomID}`, {
            exception,
            exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
        }));
    }
    /**
     * Make a user administrator via the Synapse Admin API
     * @param roomId the room where the user (or the bot) shall be made administrator.
     * @param userId optionally specify the user mxID to be made administrator.
     */
    async makeUserRoomAdmin(roomID, userID) {
        const endpoint = `/_synapse/admin/v1/rooms/${encodeURIComponent(roomID)}/make_room_admin`;
        return await this.client
            .doRequest('POST', endpoint, null, {
            user_id: userID,
        })
            .then((_) => (0, matrix_protection_suite_1.Ok)(undefined), (exception) => matrix_protection_suite_1.ActionException.Result(`Unable to make the user ${userID} admin in room ${roomID}`, {
            exception,
            exceptionKind: matrix_protection_suite_1.ActionExceptionKind.Unknown,
        }));
    }
    async getAbuseReports({ from, direction, limit, } = {}) {
        const endpoint = '/_synapse/admin/v1/event_reports';
        const queryParams = {
            ...(from ? { from } : {}),
            ...(direction ? { dir: direction } : {}),
            ...(limit ? { limit } : {}),
        };
        const response = await this.client
            .doRequest('GET', endpoint, queryParams)
            .then((value) => (0, matrix_protection_suite_1.Ok)(value), BotSDKBaseClient_1.resultifyBotSDKRequestError);
        if ((0, matrix_protection_suite_1.isError)(response)) {
            return response;
        }
        return matrix_protection_suite_1.Value.Decode(ReportPollResponse, response.ok);
    }
    async listRooms(options) {
        const endpoint = '/_synapse/admin/v1/rooms';
        return await this.client
            .doRequest('GET', endpoint, options)
            .then((response) => matrix_protection_suite_1.Value.Decode(RoomListEndpoint_1.RoomListResponse, response), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async shutdownRoomV2(roomID, options) {
        const endpoint = `/_synapse/admin/v2/rooms/${encodeURIComponent(roomID)}`;
        return await this.client
            .doRequest('DELETE', endpoint, null, options)
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async getBlockStatus(roomID) {
        const endpoint = `/_synapse/admin/v1/rooms/${encodeURIComponent(roomID)}/block`;
        return await this.client
            .doRequest('GET', endpoint)
            .then((value) => matrix_protection_suite_1.Value.Decode(BlockStatusEndpoint_1.BlockStatusResponse, value), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async getRoomDetails(roomID) {
        const endpoint = `/_synapse/admin/v1/rooms/${encodeURIComponent(roomID)}`;
        return await this.client.doRequest('GET', endpoint).then((value) => {
            return matrix_protection_suite_1.Value.Decode(RoomDetailsEndpoint_1.RoomDetailsResponse, value);
        }, BotSDKBaseClient_1.resultifyBotSDKRequestErrorWith404AsUndefined);
    }
    async suspendUser(userID) {
        const endpoint = `/_synapse/admin/v1/suspend/${encodeURIComponent(userID)}`;
        return await this.client
            .doRequest('PUT', endpoint, null, { suspend: true })
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async unsuspendUser(userID) {
        const endpoint = `/_synapse/admin/v1/suspend/${encodeURIComponent(userID)}`;
        return await this.client
            .doRequest('PUT', endpoint, null, { suspend: false })
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async getUserDetails(userID) {
        const endpoint = `/_synapse/admin/v2/users/${encodeURIComponent(userID)}`;
        return await this.client.doRequest('GET', endpoint).then((value) => {
            return matrix_protection_suite_1.Value.Decode(UserDetailsEndpoint_1.UserDetailsResponse, value);
        }, BotSDKBaseClient_1.resultifyBotSDKRequestErrorWith404AsUndefined);
    }
    async redactUser(userID) {
        const endpoint = `/_synapse/admin/v1/user/${encodeURIComponent(userID)}/redact`;
        return await this.client
            .doRequest('POST', endpoint, null, { rooms: [] })
            .then((value) => {
            return matrix_protection_suite_1.Value.Decode(UserRedactionEndpoint_1.UserRedactionResponse, value);
        }, BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async getUserRedactionStatus(redactionID) {
        const endpoint = `/_synapse/admin/v1/user/redact_status/${encodeURIComponent(redactionID)}`;
        return await this.client.doRequest('GET', endpoint).then((value) => {
            return matrix_protection_suite_1.Value.Decode(UserRedactionEndpoint_1.UserRedactionStatusResponse, value);
        }, BotSDKBaseClient_1.resultifyBotSDKRequestErrorWith404AsUndefined);
    }
    async shadowBanUser(userID) {
        const endpoint = `/_synapse/admin/v1/users/${encodeURIComponent(userID)}/shadow_ban`;
        return await this.client
            .doRequest('POST', endpoint, null, {})
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async unshadowBanUser(userID) {
        const endpoint = `/_synapse/admin/v1/users/${encodeURIComponent(userID)}/shadow_ban`;
        return await this.client
            .doRequest('DELETE', endpoint, null, {})
            .then(() => (0, matrix_protection_suite_1.Ok)(undefined), BotSDKBaseClient_1.resultifyBotSDKRequestError);
    }
    async unrestrictUser(userID) {
        const details = await this.getUserDetails(userID);
        if ((0, matrix_protection_suite_1.isError)(details)) {
            return details;
        }
        else if (details.ok === undefined) {
            return typescript_result_1.ResultError.Result(`Synapse cannot find details for the user ${userID}`);
        }
        else if (details.ok.shadow_banned) {
            const result = await this.unshadowBanUser(userID);
            if ((0, matrix_protection_suite_1.isError)(result)) {
                return result;
            }
            else {
                return (0, matrix_protection_suite_1.Ok)(AccountRestriction.ShadowBanned);
            }
        }
        else if (details.ok.suspended) {
            const result = await this.unsuspendUser(userID);
            if ((0, matrix_protection_suite_1.isError)(result)) {
                return result;
            }
            else {
                return (0, matrix_protection_suite_1.Ok)(AccountRestriction.Suspended);
            }
        }
        else if (details.ok.locked) {
            return typescript_result_1.ResultError.Result(`We don't support locking users yet`);
        }
        else if (details.ok.deactivated) {
            return typescript_result_1.ResultError.Result(`We can't reactivate users`);
        }
        else {
            return typescript_result_1.ResultError.Result(`
        User is already unrestricted`);
        }
    }
}
exports.SynapseAdminClient = SynapseAdminClient;
//# sourceMappingURL=SynapseAdminClient.js.map