"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
const typebox_1 = require("@sinclair/typebox");
const StringlyTypedMatrix_1 = require("../MatrixTypes/StringlyTypedMatrix");
const describeConfig_1 = require("./describeConfig");
const typescript_result_1 = require("@gnuxie/typescript-result");
const ConfigMirror_1 = require("./ConfigMirror");
const ConfigParseError_1 = require("./ConfigParseError");
const TrustedReportersConfigDescription = (0, describeConfig_1.describeConfig)({
    schema: typebox_1.Type.Object({
        mxids: typebox_1.Type.Array(StringlyTypedMatrix_1.StringUserIDSchema, { default: [], uniqueItems: true }),
        alertThreshold: typebox_1.Type.Number({ default: -1 }),
    }, { additionalProperties: false }),
});
it('Works i guess', function () {
    var _a;
    expect(TrustedReportersConfigDescription.parseConfig({
        mxids: ['@example:localhost'],
        alertThreshold: 5,
    }).isOkay).toBe(true);
    const result = TrustedReportersConfigDescription.parseConfig({
        mxids: [':something that makes no sense'],
    });
    if ((0, typescript_result_1.isOk)(result)) {
        throw new Error('Expected this to fail');
    }
    expect(result.error.errors.length).toBe(1);
    expect((_a = result.error.errors[0]) === null || _a === void 0 ? void 0 : _a.path).toBe('/mxids/0');
    expect(TrustedReportersConfigDescription.properties().length).toBe(2);
});
it('Is possible to create a mirror about a config', function () {
    const config = TrustedReportersConfigDescription.parseConfig({
        mxids: ['@example:localhost'],
    }).expect('Config should have parsed');
    const mirror = new ConfigMirror_1.StandardConfigMirror(TrustedReportersConfigDescription);
    const newConfig = mirror
        .addItem(config, 'mxids', '@alice:localhost')
        .expect('Should have added');
    expect(newConfig.mxids.length).toBe(2);
});
it('Is possible to get validation errors for adding garbage values', function () {
    const config = TrustedReportersConfigDescription.parseConfig({
        mxids: ['@example:localhost'],
    }).expect('Config should have parsed');
    const mirror = new ConfigMirror_1.StandardConfigMirror(TrustedReportersConfigDescription);
    const result = mirror.addItem(config, 'mxids', 'garbage hehe');
    if ((0, typescript_result_1.isOk)(result)) {
        throw new Error('Expected this to fail');
    }
    expect(result.error.path).toBe('/mxids/1');
    expect(result.error.diagnosis).toBe(ConfigParseError_1.ConfigErrorDiagnosis.ProblematicArrayItem);
});
it('Is possible to get correct paths from wrong values', function () {
    const config = TrustedReportersConfigDescription.parseConfig({
        mxids: ['@example:localhost'],
    }).expect('Config should have parsed');
    const numberResult = TrustedReportersConfigDescription.toMirror().setValue(config, 'alertThreshold', 'not a number');
    if ((0, typescript_result_1.isOk)(numberResult)) {
        throw new Error('Expected this to fail');
    }
    expect(numberResult.error.path).toBe('/alertThreshold');
});
it('Gives us accurate information on arrays being replaced with non-arrays, and that the config is recoverable', function () {
    var _a;
    const result = TrustedReportersConfigDescription.parseConfig({
        mxids: 'cheese wheels',
    });
    if ((0, typescript_result_1.isOk)(result)) {
        throw new Error('Expected this to fail');
    }
    expect((_a = result.error.errors[0]) === null || _a === void 0 ? void 0 : _a.diagnosis).toBe(ConfigParseError_1.ConfigErrorDiagnosis.ProblematicValue);
});
it('What happens when we just provide the completely wrong config', function () {
    const config = {
        oranges: ['moldy'],
        apples: 5,
    };
    const result = TrustedReportersConfigDescription.parseConfig(config);
    if ((0, typescript_result_1.isOk)(result)) {
        throw new Error('Expected this to fail');
    }
});
//# sourceMappingURL=describeConfig.test.js.map