import { ActionResult } from './Action';
export type CreateInstanceFromKey<K, V, AdditionalCreationArguments extends unknown[]> = (key: K, ...args: AdditionalCreationArguments) => Promise<ActionResult<V>>;
/**
 * This is a utility for any hash table that needs to create new values
 * from a `key`. The value will then be stored in the table and returned
 * each time the factory is then queried for that key.
 * This is mostly useful for singletons.
 * @typeParam AdditionalCreationArguments These are arguments that need to be
 * given to `createInstanceFromKey` when `getInstance` is called should a new
 * instance need to be created. Usually this would be some context like a matrix
 * client that can be used to fetch information.
 */
export declare class InternedInstanceFactory<K, V, AdditionalCreationArguments extends unknown[]> {
    private readonly createInstanceFromKey;
    private readonly instances;
    /**
     * If `getInstance` is called concurrently before the factory method that
     * creates the instance has finished, then the factory method could be called
     * multiple times concurrently. To prevent this, we use this map to lock
     * per key when the factory is called.
     */
    private readonly factoryLock;
    /**
     * Constructs the `InternedInstanceFactory`.
     * @param createInstanceFromKey A callable that will create new instances
     * from a key if the table doesn't have an entry for that key.
     */
    constructor(createInstanceFromKey: CreateInstanceFromKey<K, V, AdditionalCreationArguments>);
    /**
     * Find an instance associated with the key.
     * @param key The key.
     * @param args Any arguments that need to be given to `createInstanceFromKey`
     * that was provided the constructor for `InternedInstanceFactory`.
     * @returns An associated instance for the key.
     */
    getInstance(key: K, ...args: AdditionalCreationArguments): Promise<ActionResult<V>>;
    hasInstance(key: K): boolean;
    getStoredInstance(key: K): V | undefined;
    allInstances(): V[];
}
//# sourceMappingURL=InternedInstanceFactory.d.ts.map