"use strict";
// Copyright (C) 2023 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.InternedInstanceFactory = void 0;
const Action_1 = require("./Action");
/**
 * This is a utility for any hash table that needs to create new values
 * from a `key`. The value will then be stored in the table and returned
 * each time the factory is then queried for that key.
 * This is mostly useful for singletons.
 * @typeParam AdditionalCreationArguments These are arguments that need to be
 * given to `createInstanceFromKey` when `getInstance` is called should a new
 * instance need to be created. Usually this would be some context like a matrix
 * client that can be used to fetch information.
 */
class InternedInstanceFactory {
    /**
     * Constructs the `InternedInstanceFactory`.
     * @param createInstanceFromKey A callable that will create new instances
     * from a key if the table doesn't have an entry for that key.
     */
    constructor(createInstanceFromKey) {
        this.createInstanceFromKey = createInstanceFromKey;
        this.instances = new Map();
        /**
         * If `getInstance` is called concurrently before the factory method that
         * creates the instance has finished, then the factory method could be called
         * multiple times concurrently. To prevent this, we use this map to lock
         * per key when the factory is called.
         */
        this.factoryLock = new Map();
        // nothing to do.
    }
    /**
     * Find an instance associated with the key.
     * @param key The key.
     * @param args Any arguments that need to be given to `createInstanceFromKey`
     * that was provided the constructor for `InternedInstanceFactory`.
     * @returns An associated instance for the key.
     */
    async getInstance(key, ...args) {
        const instance = this.instances.get(key);
        if (instance !== undefined) {
            return (0, Action_1.Ok)(instance);
        }
        const lock = this.factoryLock.get(key);
        if (lock === undefined) {
            try {
                const factoryCallPromise = this.createInstanceFromKey(key, ...args);
                this.factoryLock.set(key, factoryCallPromise);
                const initialInstanceResult = await factoryCallPromise;
                if ((0, Action_1.isOk)(initialInstanceResult)) {
                    this.instances.set(key, initialInstanceResult.ok);
                }
                return initialInstanceResult;
            }
            finally {
                this.factoryLock.delete(key);
            }
        }
        else {
            return await lock;
        }
    }
    hasInstance(key) {
        return this.instances.has(key);
    }
    getStoredInstance(key) {
        return this.instances.get(key);
    }
    allInstances() {
        return [...this.instances.values()];
    }
}
exports.InternedInstanceFactory = InternedInstanceFactory;
//# sourceMappingURL=InternedInstanceFactory.js.map