"use strict";
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyRuleEvent = exports.PolicyRuleRoom = exports.PolicyRuleServer = exports.PolicyRuleUser = exports.GeneircPolicyRuleEvent = exports.RedactablePolicyContent = exports.UnredactedPolicyContent = exports.HashedPolicyContent = exports.PlainTextPolicyContent = exports.ALL_RULE_TYPES = exports.SERVER_RULE_TYPES = exports.ROOM_RULE_TYPES = exports.USER_RULE_TYPES = exports.PolicyRuleType = void 0;
exports.variantsForPolicyRuleType = variantsForPolicyRuleType;
exports.normalisePolicyRuleType = normalisePolicyRuleType;
exports.isPolicyTypeObsolete = isPolicyTypeObsolete;
exports.isPolicyRuleEvent = isPolicyRuleEvent;
const typebox_1 = require("@sinclair/typebox");
const Events_1 = require("./Events");
const Value_1 = require("../Interface/Value");
var PolicyRuleType;
(function (PolicyRuleType) {
    /// `entity` is to be parsed as a glob of users IDs
    PolicyRuleType["User"] = "m.policy.rule.user";
    /// `entity` is to be parsed as a glob of room IDs/aliases
    PolicyRuleType["Room"] = "m.policy.rule.room";
    /// `entity` is to be parsed as a glob of server names
    PolicyRuleType["Server"] = "m.policy.rule.server";
    PolicyRuleType["Unknown"] = "unknown";
})(PolicyRuleType || (exports.PolicyRuleType = PolicyRuleType = {}));
// FIXME: I don't see how this is relevant. The obsoleting behavior is weird,
//        and they should just be seperate rules.
// README! The order here matters for determining whether a type is obsolete, most recent should be first.
// These are the current and historical types for each type of rule which were used while MSC2313 was being developed
// and were left as an artifact for some time afterwards.
// Most rules (as of writing) will have the prefix `m.room.rule.*` as this has been in use for roughly 2 years.
exports.USER_RULE_TYPES = [
    PolicyRuleType.User,
    'm.room.rule.user',
    'org.matrix.mjolnir.rule.user',
];
exports.ROOM_RULE_TYPES = [
    PolicyRuleType.Room,
    'm.room.rule.room',
    'org.matrix.mjolnir.rule.room',
];
exports.SERVER_RULE_TYPES = [
    PolicyRuleType.Server,
    'm.room.rule.server',
    'org.matrix.mjolnir.rule.server',
];
exports.ALL_RULE_TYPES = [
    ...exports.USER_RULE_TYPES,
    ...exports.ROOM_RULE_TYPES,
    ...exports.SERVER_RULE_TYPES,
];
function variantsForPolicyRuleType(type) {
    switch (type) {
        case PolicyRuleType.User:
            return exports.USER_RULE_TYPES;
        case PolicyRuleType.Server:
            return exports.SERVER_RULE_TYPES;
        case PolicyRuleType.Room:
            return exports.ROOM_RULE_TYPES;
        default:
            throw new TypeError(`Unknown policy rule type ${type}`);
    }
}
function normalisePolicyRuleType(type) {
    if (exports.USER_RULE_TYPES.includes(type)) {
        return PolicyRuleType.User;
    }
    else if (exports.SERVER_RULE_TYPES.includes(type)) {
        return PolicyRuleType.Server;
    }
    else if (exports.ROOM_RULE_TYPES.includes(type)) {
        return PolicyRuleType.Room;
    }
    else {
        return PolicyRuleType.Unknown; // invalid/unknown
    }
}
function isPolicyTypeObsolete(normalisedType, existingType, candidateType) {
    switch (normalisedType) {
        case PolicyRuleType.User:
            return (exports.USER_RULE_TYPES.indexOf(candidateType) >
                exports.USER_RULE_TYPES.indexOf(existingType));
        case PolicyRuleType.Server:
            return (exports.SERVER_RULE_TYPES.indexOf(candidateType) >
                exports.SERVER_RULE_TYPES.indexOf(existingType));
        case PolicyRuleType.Room:
            return (exports.ROOM_RULE_TYPES.indexOf(candidateType) >
                exports.ROOM_RULE_TYPES.indexOf(existingType));
        default:
            throw new TypeError(`unknown PolicyRuleType ${normalisedType}`);
    }
}
exports.PlainTextPolicyContent = typebox_1.Type.Object({
    entity: typebox_1.Type.String({
        description: 'The entity affected by this rule. Glob characters `*` and `?` can be used\nto match zero or more characters or exactly one character respectively.',
    }),
    recommendation: typebox_1.Type.String({
        description: 'The suggested action to take. Currently only `m.ban` is specified.',
    }),
    reason: typebox_1.Type.Optional(typebox_1.Type.String({
        description: 'The human-readable description for the `recommendation`.',
    })),
});
exports.HashedPolicyContent = typebox_1.Type.Union([
    typebox_1.Type.Intersect([
        typebox_1.Type.Omit(exports.PlainTextPolicyContent, ['entity']),
        typebox_1.Type.Object({
            'org.matrix.msc4205.hashes': typebox_1.Type.Record(typebox_1.Type.String(), typebox_1.Type.String()),
        }),
    ]),
    typebox_1.Type.Intersect([
        typebox_1.Type.Omit(exports.PlainTextPolicyContent, ['entity']),
        typebox_1.Type.Object({
            hashes: typebox_1.Type.Record(typebox_1.Type.String(), typebox_1.Type.String()),
        }),
    ]),
]);
exports.UnredactedPolicyContent = typebox_1.Type.Union([
    typebox_1.Type.Union([exports.PlainTextPolicyContent, exports.HashedPolicyContent]),
]);
exports.RedactablePolicyContent = typebox_1.Type.Union([
    exports.UnredactedPolicyContent,
    Events_1.EmptyContent,
]);
exports.GeneircPolicyRuleEvent = (0, Events_1.StateEvent)(exports.RedactablePolicyContent);
exports.PolicyRuleUser = typebox_1.Type.Intersect([
    exports.GeneircPolicyRuleEvent,
    typebox_1.Type.Object({
        state_key: typebox_1.Type.Optional(typebox_1.Type.String({
            description: 'An arbitrary string decided upon by the sender.',
        })),
        type: typebox_1.Type.Union(exports.USER_RULE_TYPES.map((type) => typebox_1.Type.Literal(type))),
    }),
]);
exports.PolicyRuleServer = typebox_1.Type.Intersect([
    exports.GeneircPolicyRuleEvent,
    typebox_1.Type.Object({
        state_key: typebox_1.Type.Optional(typebox_1.Type.String({
            description: 'An arbitrary string decided upon by the sender.',
        })),
        type: typebox_1.Type.Union(exports.SERVER_RULE_TYPES.map((type) => typebox_1.Type.Literal(type))),
    }),
]);
exports.PolicyRuleRoom = typebox_1.Type.Intersect([
    exports.GeneircPolicyRuleEvent,
    typebox_1.Type.Object({
        state_key: typebox_1.Type.Optional(typebox_1.Type.String({
            description: 'An arbitrary string decided upon by the sender.',
        })),
        type: typebox_1.Type.Union(exports.ROOM_RULE_TYPES.map((type) => typebox_1.Type.Literal(type))),
    }),
]);
exports.PolicyRuleEvent = typebox_1.Type.Union([
    exports.PolicyRuleUser,
    exports.PolicyRuleServer,
    exports.PolicyRuleRoom,
]);
Value_1.Value.Compile(exports.PolicyRuleEvent);
function isPolicyRuleEvent(value) {
    return Value_1.Value.Check(exports.PolicyRuleEvent, value);
}
//# sourceMappingURL=PolicyEvents.js.map