import { PolicyRuleType } from '../MatrixTypes/PolicyEvents';
import { EntityMatchOptions, PolicyListRevision } from './PolicyListRevision';
import { EntityPolicyRule, GlobPolicyRule, HashedLiteralPolicyRule, LiteralPolicyRule, PolicyRule, Recommendation } from './PolicyRule';
import { PolicyRuleChange } from './PolicyRuleChange';
import { Revision } from './Revision';
import { Map as PersistentMap, List as PersistentList } from 'immutable';
import { StringEventID } from '@the-draupnir-project/matrix-basic-types';
/**
 * A map of policy rules, by their type and then event id.
 */
type PolicyRuleByType = PersistentMap<PolicyRuleType, PersistentMap<StringEventID, PolicyRule>>;
type PolicyRuleScopes = PersistentMap<PolicyRuleType, PersistentMap<Recommendation, PolicyRuleScope>>;
/**
 * A standard implementation of a `PolicyListRevision` using immutable's persistent maps.
 */
export declare class StandardPolicyListRevision implements PolicyListRevision {
    readonly revisionID: Revision;
    /**
     * Allow us to detect whether we have updated the state for this event.
     */
    private readonly policyRuleByType;
    private readonly policyRuleScopes;
    /**
     * Use {@link StandardPolicyListRevision.blankRevision} to get started.
     * Only use this constructor if you are implementing a variant of PolicyListRevision.
     * @param revisionID A revision ID to represent this revision.
     * @param policyRules A map containing the rules for this revision by state type and then state key.
     * @param policyRuleByEventId A map containing the rules ofr this revision by event id.
     */
    constructor(revisionID: Revision, 
    /**
     * Allow us to detect whether we have updated the state for this event.
     */
    policyRuleByType: PolicyRuleByType, policyRuleScopes: PolicyRuleScopes);
    /**
     * @returns An empty revision.
     */
    static blankRevision(): StandardPolicyListRevision;
    isBlankRevision(): boolean;
    allRules(): PolicyRule[];
    allRulesMatchingEntity(entity: string, { recommendation, type: ruleKind, searchHashedRules, }: Partial<EntityMatchOptions>): PolicyRule[];
    findRulesMatchingHash(hash: string, algorithm: string, { type, recommendation, }: Partial<Pick<EntityMatchOptions, 'recommendation'>> & Pick<EntityMatchOptions, 'type'>): HashedLiteralPolicyRule[];
    findRuleMatchingEntity(entity: string, { recommendation, type, searchHashedRules }: EntityMatchOptions): PolicyRule | undefined;
    allRulesOfType(type: PolicyRuleType, recommendation?: Recommendation): PolicyRule[];
    reviseFromChanges(changes: PolicyRuleChange[]): StandardPolicyListRevision;
    hasEvent(eventId: string): boolean;
    hasPolicy(eventID: StringEventID): boolean;
    getPolicy(eventID: StringEventID): PolicyRule | undefined;
}
export type PolicyRuleChangeByScope = Map<PolicyRuleType, Map<Recommendation, PolicyRuleChange[]>>;
export declare function groupChangesByScope(changes: PolicyRuleChange[]): PolicyRuleChangeByScope;
type PolicyRuleByEntity<Rule extends EntityPolicyRule = EntityPolicyRule> = PersistentMap<string, PersistentList<Rule>>;
/**
 * A scope is a collection of rules that are scoped to a single entity type and
 * recommendation. So for the most basic policy list, there will usually be
 * a scope for all the `m.policy.rule.user` events that have the recommendation
 * `m.ban`.
 *
 * Scopes are built, quite painfully, to make rule lookup convienant and quick.
 * We accept this because revisions are few and far between, and if they are
 * frequent, will have a very small number of change events.
 */
declare class PolicyRuleScope {
    readonly revisionID: Revision;
    /**
     * The entity type that this cache is for e.g. RULE_USER.
     */
    readonly entityType: PolicyRuleType;
    /**
     * The recommendation that this cache is for e.g. m.ban (RECOMMENDATION_BAN).
     */
    readonly recommendation: Recommendation;
    /**
     * Glob rules always have to be scanned against every entity.
     */
    private readonly globRules;
    /**
     * This table allows us to skip matching an entity against every literal.
     */
    private readonly literalRules;
    /**
     * Hashed literal rules. This tables allows us to quickly find hashed rules.
     */
    private readonly sha256HashedLiteralRules;
    static blankScope(revisionID: Revision, ruleType: PolicyRuleType, recommendation: Recommendation): PolicyRuleScope;
    constructor(revisionID: Revision, 
    /**
     * The entity type that this cache is for e.g. RULE_USER.
     */
    entityType: PolicyRuleType, 
    /**
     * The recommendation that this cache is for e.g. m.ban (RECOMMENDATION_BAN).
     */
    recommendation: Recommendation, 
    /**
     * Glob rules always have to be scanned against every entity.
     */
    globRules: PolicyRuleByEntity<GlobPolicyRule>, 
    /**
     * This table allows us to skip matching an entity against every literal.
     */
    literalRules: PolicyRuleByEntity<LiteralPolicyRule>, 
    /**
     * Hashed literal rules. This tables allows us to quickly find hashed rules.
     */
    sha256HashedLiteralRules: PersistentMap<string, PersistentList<HashedLiteralPolicyRule>>);
    reviseFromChanges(revision: Revision, changes: PolicyRuleChange[], rulesByEventID: PersistentMap<StringEventID, PolicyRule>): PolicyRuleScope;
    literalRulesMatchingEntity(entity: string): PolicyRule[];
    globRulesMatchingEntity(entity: string): PolicyRule[];
    hashedRulesMatchingEntity(entity: string): PolicyRule[];
    allRulesMatchingEntity(entity: string, searchHashedRules: boolean): PolicyRule[];
    findRuleMatchingEntity(entity: string, searchHashedRules: boolean): PolicyRule | undefined;
    findHashRules(hash: string): HashedLiteralPolicyRule[];
}
export {};
//# sourceMappingURL=StandardPolicyListRevision.d.ts.map