"use strict";
// Copyright 2022 - 2023 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardPolicyRoomRevisionIssuer = void 0;
const stream_1 = require("stream");
const Action_1 = require("../Interface/Action");
const Logger_1 = require("../Logging/Logger");
const log = new Logger_1.Logger('StandardPolicyRoomRevisionIssuer');
/**
 * A standard implementation of PolicyRoomRevisionIssuer.
 */
class StandardPolicyRoomRevisionIssuer extends stream_1.EventEmitter {
    /**
     * Creates a new StandardPolicyRoomRevisionIssuer, you shouldn't have to use this,
     * instead use the `PolicyRoomManager`.
     * @see {@link PolicyRoomManager}.
     * @param room The matrix room to issue revisions for.
     * @param currentRevision The current revision for the room, can be blank.
     * @param policyListManager The policy list manager to use to fetch room state with.
     */
    constructor(room, currentRevision, policyListManager) {
        super();
        this.room = room;
        this.currentRevision = currentRevision;
        this.batcher = new RevisionBatcher(this, policyListManager);
    }
    updateForStateEvent(event) {
        if (this.currentRevision.hasEvent(event.event_id)) {
            return;
        }
        this.batcher.addToBatch(event.event_id);
    }
    updateForRedactionEvent(event) {
        if (this.currentRevision.hasEvent(event.event_id)) {
            return;
        }
        this.batcher.addToBatch(event.event_id);
    }
    updateForRevealedPolicies(policies) {
        const changes = this.currentRevision.changesFromRevealedPolicies(policies);
        if (changes.length === 0) {
            return;
        }
        const previousRevision = this.currentRevision;
        this.currentRevision = previousRevision.reviseFromChanges(changes);
        this.emit('revision', this.currentRevision, changes, previousRevision);
    }
    unregisterListeners() {
        // nothing to do.
    }
}
exports.StandardPolicyRoomRevisionIssuer = StandardPolicyRoomRevisionIssuer;
/**
 * Helper class that emits a batch event on a `PolicyList` when it has made a batch
 * out of the Matrix events given to `addToBatch` via `updateForEvent`.
 * The `RevisionBatcher` will then call `list.update()` on the associated `PolicyList` once it has finished batching events.
 */
class RevisionBatcher {
    constructor(policyListRevisionIssuer, policyListManager) {
        this.policyListRevisionIssuer = policyListRevisionIssuer;
        this.policyListManager = policyListManager;
        // Whether we are waiting for more events to form a batch.
        this.isWaiting = false;
        // The latest (or most recent) event we have received.
        this.latestEventId = null;
        this.waitPeriodMS = 200; // 200ms seems good enough.
        this.maxWaitMS = 3000; // 3s is long enough to wait while batching.
        // Events that the batcher has been informed of
        this.batchedEvents = new Set();
    }
    /**
     * Reset the state for the next batch.
     */
    reset() {
        this.latestEventId = null;
        this.isWaiting = false;
        this.batchedEvents.clear();
    }
    /**
     * Checks if any more events have been added to the current batch since
     * the previous iteration, then keep waiting up to `this.maxWait`, otherwise stop
     * and emit a batch.
     * @param eventId The id of the first event for this batch.
     */
    async checkBatch(eventId) {
        const start = Date.now();
        do {
            await new Promise((resolve) => setTimeout(resolve, this.waitPeriodMS));
        } while (Date.now() - start < this.maxWaitMS &&
            this.latestEventId !== eventId);
        this.reset();
        // batching finished, update the associated list.
        await this.createBatchedRevision();
    }
    /**
     * Adds an event to the batch.
     * @param eventId The event to inform the batcher about.
     */
    addToBatch(eventId) {
        if (this.batchedEvents.has(eventId)) {
            return;
        }
        this.latestEventId = eventId;
        this.batchedEvents.add(eventId);
        if (this.isWaiting) {
            return;
        }
        this.isWaiting = true;
        // We 'spawn' off here after performing the checks above
        // rather than before (ie if `addToBatch` was async) because
        // `banListTest` showed that there were 100~ ACL events per protected room
        // as compared to just 5~ by doing this. Not entirely sure why but it probably
        // has to do with queuing up `n event` tasks on the event loop that exaust scheduling
        // (so the latency between them is percieved as much higher by
        // the time they get checked in `this.checkBatch`, thus batching fails).
        void this.checkBatch(eventId);
    }
    async createBatchedRevision() {
        const roomID = this.policyListRevisionIssuer.room;
        const policyRuleEventsResult = await this.policyListManager.getPolicyRuleEvents(roomID);
        if ((0, Action_1.isError)(policyRuleEventsResult)) {
            log.error(`Unable to fetch policy rule events for ${roomID.toPermalink()}, this is really bad if this error is persistent`);
            return;
        }
        const nextRevision = this.policyListRevisionIssuer.currentRevision.reviseFromState(policyRuleEventsResult.ok);
        const previousRevision = this.policyListRevisionIssuer.currentRevision;
        const changes = previousRevision.changesFromState(policyRuleEventsResult.ok);
        this.policyListRevisionIssuer.currentRevision = nextRevision;
        this.policyListRevisionIssuer.emit('revision', nextRevision, changes, previousRevision);
    }
}
//# sourceMappingURL=StandardPolicyRoomRevisionIssuer.js.map