"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
var __addDisposableResource = (this && this.__addDisposableResource) || function (env, value, async) {
    if (value !== null && value !== void 0) {
        if (typeof value !== "object" && typeof value !== "function") throw new TypeError("Object expected.");
        var dispose, inner;
        if (async) {
            if (!Symbol.asyncDispose) throw new TypeError("Symbol.asyncDispose is not defined.");
            dispose = value[Symbol.asyncDispose];
        }
        if (dispose === void 0) {
            if (!Symbol.dispose) throw new TypeError("Symbol.dispose is not defined.");
            dispose = value[Symbol.dispose];
            if (async) inner = dispose;
        }
        if (typeof dispose !== "function") throw new TypeError("Object not disposable.");
        if (inner) dispose = function() { try { inner.call(this); } catch (e) { return Promise.reject(e); } };
        env.stack.push({ value: value, dispose: dispose, async: async });
    }
    else if (async) {
        env.stack.push({ async: true });
    }
    return value;
};
var __disposeResources = (this && this.__disposeResources) || (function (SuppressedError) {
    return function (env) {
        function fail(e) {
            env.error = env.hasError ? new SuppressedError(e, env.error, "An error was suppressed during disposal.") : e;
            env.hasError = true;
        }
        function next() {
            while (env.stack.length) {
                var rec = env.stack.pop();
                try {
                    var result = rec.dispose && rec.dispose.call(rec.value);
                    if (rec.async) return Promise.resolve(result).then(next, function(e) { fail(e); return next(); });
                }
                catch (e) {
                    fail(e);
                }
            }
            if (env.hasError) throw env.error;
        }
        return next();
    };
})(typeof SuppressedError === "function" ? SuppressedError : function (error, suppressed, message) {
    var e = new Error(message);
    return e.name = "SuppressedError", e.error = error, e.suppressed = suppressed, e;
});
Object.defineProperty(exports, "__esModule", { value: true });
exports.HandleRegistrySemantics = void 0;
const typescript_result_1 = require("@gnuxie/typescript-result");
const SemanticType_1 = require("../../Interface/SemanticType");
const Lifetime_1 = require("../../Interface/Lifetime");
const HandleDescription_1 = require("./HandleDescription");
exports.HandleRegistrySemantics = (0, SemanticType_1.SemanticType)('HandleRegistry').Law({
    establishHandles: {
        what: 'When registerPluginHandles is called, plugins will later receive calls for their handles',
        why: 'Provides the hook point for plugins to register with handles',
        law: 'For plugin P and handle H, registering plugin will result in handle H being called on plugin when invoked',
        async check(makeSubject) {
            const env_1 = { stack: [], error: void 0, hasError: false };
            try {
                const description = (await makeSubject()).expect('Should be able to make the subject');
                let publishHandleCallback;
                let handleInvocations = 0;
                const testHandleDescription = {
                    handleName: 'testHandle',
                    dataSourceType: HandleDescription_1.HandleDataSourceType.Context,
                    establish(_context, callback) {
                        publishHandleCallback = callback;
                        return (0, typescript_result_1.Ok)(undefined);
                    },
                };
                const descriptionWithHandle = description.registerHandleDescription(testHandleDescription);
                const registryLifetime = __addDisposableResource(env_1, new Lifetime_1.StandardLifetime(), true);
                const testPlugin = {
                    testHandle() {
                        handleInvocations += 1;
                    },
                };
                const lifetime = __addDisposableResource(env_1, new Lifetime_1.StandardLifetime(), true);
                descriptionWithHandle
                    .registryForContext(registryLifetime, {})
                    .expect('registry creation failed')
                    .registerPluginHandles(testPlugin, lifetime);
                if (publishHandleCallback === undefined) {
                    throw new TypeError('Handle establishment did not provide a publish callback');
                }
                publishHandleCallback('testHandle');
                if (handleInvocations !== 1) {
                    throw new TypeError('Registered handle was not invoked after publish');
                }
            }
            catch (e_1) {
                env_1.error = e_1;
                env_1.hasError = true;
            }
            finally {
                const result_1 = __disposeResources(env_1);
                if (result_1)
                    await result_1;
            }
        },
    },
    pluginRemoval: {
        what: 'Handles will no longer be called on plugins that are unregistered',
        why: 'Make sure that plugins can be cleanly removed from the system',
        law: 'For plugin P and handle H, after unregistering P, H will no longer be called on P',
        async check(makeSubject) {
            const env_2 = { stack: [], error: void 0, hasError: false };
            try {
                const description = (await makeSubject()).expect('Should be able to make the subject');
                let publishHandleCallback;
                let handleInvocations = 0;
                const handle = {
                    handleName: 'handle',
                    dataSourceType: HandleDescription_1.HandleDataSourceType.Context,
                    establish(_context, publish) {
                        publishHandleCallback = publish;
                        return (0, typescript_result_1.Ok)(undefined);
                    },
                };
                const descriptionWithHandle = description.registerHandleDescription(handle);
                const registryLifetime = __addDisposableResource(env_2, new Lifetime_1.StandardLifetime(), true);
                const registry = __addDisposableResource(env_2, descriptionWithHandle
                    .registryForContext(registryLifetime, {})
                    .expect('Should be able to construct registry for context'), true);
                const plugin = {
                    handle() {
                        handleInvocations += 1;
                    },
                };
                const pluginLifetime = __addDisposableResource(env_2, new Lifetime_1.StandardLifetime(), true);
                registry
                    .registerPluginHandles(plugin, pluginLifetime)
                    .expect('Should be able to register plugin handles');
                if (publishHandleCallback === undefined) {
                    throw new TypeError('Handle establishment did not provide a publish callback');
                }
                publishHandleCallback('handle');
                registry.removePluginHandles(plugin);
                publishHandleCallback('handle');
                if (handleInvocations !== 1) {
                    throw new TypeError('Handle was invoked after plugin removal. It should not be.');
                }
            }
            catch (e_2) {
                env_2.error = e_2;
                env_2.hasError = true;
            }
            finally {
                const result_2 = __disposeResources(env_2);
                if (result_2)
                    await result_2;
            }
        },
    },
    unaryHandleRegistration: {
        what: 'Plugin registration is unary, handles will not be called multiple times as a result of multiple registration',
        why: 'Prevents bugs from multiple registration',
        law: 'For a plugin P, and handle H, calling registerHandles(P) twice will result in H of P being called exactly once only',
        async check(makeSubject) {
            const env_3 = { stack: [], error: void 0, hasError: false };
            try {
                const description = (await makeSubject()).expect('Should be able to make the subject');
                let establishCount = 0;
                const handle = {
                    handleName: 'handle',
                    dataSourceType: HandleDescription_1.HandleDataSourceType.Plugin,
                    establish: () => {
                        establishCount += 1;
                        return (0, typescript_result_1.Ok)(undefined);
                    },
                };
                const descriptionWithHandle = description.registerHandleDescription(handle);
                const registryLifetime = __addDisposableResource(env_3, new Lifetime_1.StandardLifetime(), true);
                const registry = __addDisposableResource(env_3, descriptionWithHandle
                    .registryForContext(registryLifetime, {})
                    .expect('Should be able to construct registry for context'), true);
                let handleInvocations = 0;
                const plugin = {
                    handle() {
                        handleInvocations += 1;
                    },
                };
                const pluginLifetime = __addDisposableResource(env_3, new Lifetime_1.StandardLifetime(), true);
                registry
                    .registerPluginHandles(plugin, pluginLifetime)
                    .expect('Should be able to register plugin handles');
                registry
                    .registerPluginHandles(plugin, pluginLifetime)
                    .expect('Should be able to re-register plugin handles');
                if (establishCount !== 1) {
                    throw new TypeError('Plugin handle establish should only have been called once');
                }
                plugin.handle();
                if (handleInvocations !== 1) {
                    throw new TypeError('Plugin handle should have been called once');
                }
            }
            catch (e_3) {
                env_3.error = e_3;
                env_3.hasError = true;
            }
            finally {
                const result_3 = __disposeResources(env_3);
                if (result_3)
                    await result_3;
            }
        },
    },
    disposable: {
        what: 'HandleRegistry un-registers all plugins on disposal',
        why: 'Prevents resource leaks from HandleRegistry instances',
        law: 'For plugin P and handle H, after disposing the HandleRegistry, H will no longer be called on P',
        async check(makeSubject) {
            const description = (await makeSubject()).expect('Should be able to make the subject');
            let publishHandleCallback;
            let handleInvocations = 0;
            const handle = {
                handleName: 'handle',
                dataSourceType: HandleDescription_1.HandleDataSourceType.Context,
                establish(_context, publish) {
                    publishHandleCallback = publish;
                    return (0, typescript_result_1.Ok)(undefined);
                },
            };
            const descriptionWithHandle = description.registerHandleDescription(handle);
            {
                const env_4 = { stack: [], error: void 0, hasError: false };
                try {
                    const registryLifetime = __addDisposableResource(env_4, new Lifetime_1.StandardLifetime(), true);
                    const registry = __addDisposableResource(env_4, descriptionWithHandle
                        .registryForContext(registryLifetime, {})
                        .expect('Should be able to construct registry for context'), true);
                    const plugin = {
                        handle() {
                            handleInvocations += 1;
                        },
                    };
                    const pluginLifetime = __addDisposableResource(env_4, new Lifetime_1.StandardLifetime(), true);
                    registry
                        .registerPluginHandles(plugin, pluginLifetime)
                        .expect('Should be able to register plugin handles');
                    if (publishHandleCallback === undefined) {
                        throw new TypeError('Handle establishment did not provide a publish callback');
                    }
                    publishHandleCallback('handle');
                }
                catch (e_4) {
                    env_4.error = e_4;
                    env_4.hasError = true;
                }
                finally {
                    const result_4 = __disposeResources(env_4);
                    if (result_4)
                        await result_4;
                }
            }
            try {
                publishHandleCallback('handle');
            }
            catch (_a) {
                // catch errors from invoking after disposal
            }
            if (handleInvocations !== 1) {
                throw new TypeError('Handle was invoked after registry disposal. It should not be.');
            }
        },
    },
});
//# sourceMappingURL=HandleRegistry.js.map