"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardHandleRegistry = void 0;
const HandleDescription_1 = require("./HandleDescription");
const typescript_result_1 = require("@gnuxie/typescript-result");
class StandardHandleRegistry {
    constructor(context, lifetime) {
        this.context = context;
        this.lifetime = lifetime;
        this.handleDescriptions = new Map();
        this.plugins = new Set();
        this.forwardHandleFromContext = (handleName, ...args) => {
            const handleDescription = this.getHandleDescription(handleName);
            if (handleDescription === undefined) {
                throw new TypeError(`No such handle registered: ${handleName}`);
            }
            for (const plugin of this.plugins) {
                if (handleName in plugin) {
                    const pluginHandle = plugin[handleName];
                    pluginHandle(...args);
                }
            }
        };
        // nothing to do.
    }
    getHandleDescription(name) {
        return this.handleDescriptions.get(name);
    }
    registerHandleDescription(description) {
        const establishResult = description.dataSourceType === HandleDescription_1.HandleDataSourceType.Context
            ? description.establish(this.context, this.forwardHandleFromContext, this.lifetime)
            : (0, typescript_result_1.Ok)(undefined);
        if ((0, typescript_result_1.isError)(establishResult)) {
            return establishResult;
        }
        this.handleDescriptions.set(description.handleName, description);
        return (0, typescript_result_1.Ok)(this);
    }
    registerPluginHandles(plugin, pluginLifetime) {
        if (this.plugins.has(plugin)) {
            return (0, typescript_result_1.Ok)(this);
        }
        for (const handle of this.handleDescriptions.values()) {
            if (handle.handleName in plugin) {
                if (handle.dataSourceType === HandleDescription_1.HandleDataSourceType.Plugin) {
                    const establishResult = handle.establish(this.context, plugin, pluginLifetime);
                    if ((0, typescript_result_1.isError)(establishResult)) {
                        return establishResult;
                    }
                }
            }
        }
        this.plugins.add(plugin);
        return (0, typescript_result_1.Ok)(this);
    }
    removePluginHandles(plugin) {
        this.plugins.delete(plugin);
    }
    async [Symbol.asyncDispose]() {
        await this.lifetime[Symbol.asyncDispose]();
        this.plugins.clear();
        this.handleDescriptions.clear();
    }
}
exports.StandardHandleRegistry = StandardHandleRegistry;
//# sourceMappingURL=StandardHandleRegistry.js.map