"use strict";
// SPDX-FileCopyrightText: 2023-2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardProtectedRoomsSet = void 0;
const Task_1 = require("../Interface/Task");
const PowerLevelsMirror_1 = require("../Client/PowerLevelsMirror");
const ProtectedRoomsManager_1 = require("./ProtectedRoomsManager/ProtectedRoomsManager");
const SetMembershipPolicyRevisionIssuer_1 = require("../MembershipPolicies/SetMembershipPolicyRevisionIssuer");
const CreateRoom_1 = require("../MatrixTypes/CreateRoom");
class StandardProtectedRoomsSet {
    constructor(watchedPolicyRooms, protectedRoomsManager, protections, userID, eventMixinExtractor, handleMissingProtectionPermissions) {
        this.watchedPolicyRooms = watchedPolicyRooms;
        this.protectedRoomsManager = protectedRoomsManager;
        this.protections = protections;
        this.userID = userID;
        this.eventMixinExtractor = eventMixinExtractor;
        this.handleMissingProtectionPermissions = handleMissingProtectionPermissions;
        this.membershipChangeListener = this.setMembershipChangeListener.bind(this);
        this.policyChangeListener = this.policyRevisionChangeListener.bind(this);
        this.stateChangeListener = this.stateRevisionChangeListener.bind(this);
        this.roomsChangeListener = this.protectedRoomsChangeListener.bind(this);
        this.setMembershiprevisionListener = this.setMembershipRevision.bind(this);
        this.setMembershipPolicyRevisionListener = this.setMembershipPolicyRevision.bind(this);
        this.setRoomMembership.on('membership', this.membershipChangeListener);
        this.setRoomState.on('revision', this.stateChangeListener);
        watchedPolicyRooms.revisionIssuer.on('revision', this.policyChangeListener);
        this.protectedRoomsManager.on('change', this.roomsChangeListener);
        this.setMembership.on('revision', this.setMembershiprevisionListener);
        this.setPoliciesMatchingMembership =
            new SetMembershipPolicyRevisionIssuer_1.StandardMembershipPolicyRevisionIssuer(this.setMembership, watchedPolicyRooms.revisionIssuer);
        this.setPoliciesMatchingMembership.on('revision', this.setMembershipPolicyRevisionListener);
    }
    get setRoomState() {
        return this.protectedRoomsManager.setRoomState;
    }
    get setRoomMembership() {
        return this.protectedRoomsManager.setRoomMembership;
    }
    get setMembership() {
        return this.protectedRoomsManager.setMembership;
    }
    get allProtectedRooms() {
        return this.protectedRoomsManager.allProtectedRooms;
    }
    handleTimelineEvent(roomID, event) {
        var _a;
        // this should only be responsible for passing through to protections.
        // The RoomMembershipManage (and its dependants)
        // The PolicyListManager (and its dependents)
        // The RoomStateManager (and its dependents)
        // should get informed directly elsewhere, since there's no reason
        // they cannot be shared across protected rooms sets.
        // The only slightly dodgy thing about that is the PolicyListManager
        // can depend on the RoomStateManager but i don't suppose it'll matter
        // they both are programmed to de-duplicate repeat events.
        const room = this.protectedRoomsManager.getProtectedRoom(roomID);
        if (room === undefined) {
            throw new TypeError(`The protected rooms set should not be being informed about events that it is not protecting`);
        }
        const mixinEvent = this.eventMixinExtractor.parseEvent(event);
        for (const protection of this.protections.allProtections) {
            if (protection.handleTimelineEvent !== undefined) {
                void (0, Task_1.Task)(protection.handleTimelineEvent(room, event));
            }
            (_a = protection.handleTimelineEventMixins) === null || _a === void 0 ? void 0 : _a.call(protection, room, mixinEvent);
        }
    }
    handleEventReport(report) {
        for (const protection of this.protections.allProtections) {
            if (protection.handleEventReport === undefined) {
                continue;
            }
            void (0, Task_1.Task)(protection.handleEventReport(report));
        }
    }
    handleExternalMembership(roomID, event) {
        for (const protection of this.protections.allProtections) {
            if (protection.handleExternalMembership === undefined) {
                continue;
            }
            protection.handleExternalMembership(roomID, event);
        }
    }
    isProtectedRoom(roomID) {
        return this.protectedRoomsManager.isProtectedRoom(roomID);
    }
    setMembershipChangeListener(_roomID, nextRevision, changes, _previousRevision) {
        for (const protection of this.protections.allProtections) {
            if (protection.handleMembershipChange === undefined) {
                continue;
            }
            void (0, Task_1.Task)(protection.handleMembershipChange(nextRevision, changes));
        }
    }
    policyRevisionChangeListener(nextRevision, changes, _previousRevision) {
        for (const protection of this.protections.allProtections) {
            if (protection.handlePolicyChange === undefined) {
                continue;
            }
            void (0, Task_1.Task)(protection.handlePolicyChange(nextRevision, changes));
        }
    }
    /**
     * To be called only after power levels
     * have changed in a room. For some reason it's conflicted with
     * checking permissions within a room that has just been added.
     * Which is wrong.
     *
     * I really don't know how to fix this right now!!
     */
    powerLevelsChangeFromContent(room, createEvent, isNewlyAddedRoom, nextPowerLevels, previousPowerLevels) {
        var _a, _b;
        // prividliged creators never change and always have permission.
        if (CreateRoom_1.RoomVersionMirror.isUserAPrivilidgedCreator(this.userID, createEvent)) {
            return;
        }
        const missingPermissionsInfo = [];
        for (const protection of this.protections.allProtections) {
            const permissionsChange = PowerLevelsMirror_1.PowerLevelsMirror.calculateNewMissingPermissions(this.userID, {
                nextPowerLevelsContent: nextPowerLevels !== null && nextPowerLevels !== void 0 ? nextPowerLevels : {},
                previousPowerLevelsContent: previousPowerLevels !== null && previousPowerLevels !== void 0 ? previousPowerLevels : {},
                requiredEventPermissions: protection.requiredEventPermissions,
                requiredPermissions: protection.requiredPermissions,
                requiredStatePermissions: protection.requiredStatePermissions,
                createEvent,
                isNewlyAddedRoom,
            });
            const { isPrivilidgedInNextPowerLevels, isPrivilidgedInPriorPowerLevels, } = permissionsChange;
            if (!isPrivilidgedInNextPowerLevels) {
                missingPermissionsInfo.push({
                    protection,
                    permissionsChange,
                });
            }
            if (isPrivilidgedInNextPowerLevels && !isPrivilidgedInPriorPowerLevels) {
                (_a = protection.handlePermissionRequirementsMet) === null || _a === void 0 ? void 0 : _a.call(protection, room);
            }
        }
        if (missingPermissionsInfo.length !== 0) {
            (_b = this.handleMissingProtectionPermissions) === null || _b === void 0 ? void 0 : _b.call(this, room.toRoomIDOrAlias(), missingPermissionsInfo);
        }
    }
    powerLevelsChangeFromRevision(nextRevision, previousRevision) {
        const previousPowerLevels = previousRevision.getStateEvent('m.room.power_levels', '');
        const nextPowerLevels = nextRevision.getStateEvent('m.room.power_levels', '');
        const createEvent = nextRevision.getStateEvent('m.room.create', '');
        if (createEvent === undefined) {
            throw new TypeError('Room with missing create event found, this is not ok');
        }
        this.powerLevelsChangeFromContent(nextRevision.room, createEvent, false, nextPowerLevels === null || nextPowerLevels === void 0 ? void 0 : nextPowerLevels.content, previousPowerLevels === null || previousPowerLevels === void 0 ? void 0 : previousPowerLevels.content);
    }
    stateRevisionChangeListener(_roomID, nextRevision, changes, previousRevision) {
        const powerLevelsEvent = changes.find((change) => change.eventType === 'm.room.power_levels');
        if (powerLevelsEvent !== undefined) {
            this.powerLevelsChangeFromRevision(nextRevision, previousRevision);
        }
        for (const protection of this.protections.allProtections) {
            if (protection.handleStateChange === undefined) {
                continue;
            }
            void (0, Task_1.Task)(protection.handleStateChange(nextRevision, changes));
        }
    }
    protectedRoomsChangeListener(room, changeType) {
        if (changeType !== ProtectedRoomsManager_1.ProtectedRoomChangeType.Added) {
            return;
        }
        const currentRevision = this.setRoomState.getRevision(room.toRoomIDOrAlias());
        if (currentRevision === undefined) {
            throw new TypeError(`The SetRoomState is not being kept consistent with the number of protected rooms`);
        }
        const currentPowerLevelsEvent = currentRevision.getStateEvent('m.room.power_levels', '');
        const createEvent = currentRevision.getStateEvent('m.room.create', '');
        if (createEvent === undefined) {
            throw new TypeError('Room with missing create event found, this is not ok');
        }
        // We call the powerLevelsChange so that handlePermissionsMet will be called
        // on protections for with the new room.
        // We also call powerLevelsChange so that the missing permissions CB will
        // get called if we don't have the right permissions for any protection in the new room.
        this.powerLevelsChangeFromContent(room, createEvent, true, currentPowerLevelsEvent === null || currentPowerLevelsEvent === void 0 ? void 0 : currentPowerLevelsEvent.content, 
        // always treat the previous revision as though we are unprividdged in the new room.
        {
            users_default: -1,
        });
    }
    setMembershipRevision(nextRevision, changes) {
        for (const protection of this.protections.allProtections) {
            if (protection.handleSetMembershipChange !== undefined) {
                protection.handleSetMembershipChange(nextRevision, changes);
            }
        }
    }
    setMembershipPolicyRevision(nextRevision, changes) {
        for (const protection of this.protections.allProtections) {
            if (protection.handleSetMembershipPolicyMatchesChange !== undefined) {
                protection.handleSetMembershipPolicyMatchesChange(nextRevision, changes);
            }
        }
    }
    unregisterListeners() {
        // The most important listenres to reach is the setRoomState, setRoommMembership,
        // and policy revision listeners. Since these are tied to the global and
        // shared revision issuers that are given by the "RoomStateManager" deriratives.
        // The listener situation here kinda sucks, setting up and managing these relationships
        // should be left to some other component.
        this.setRoomMembership.off('membership', this.membershipChangeListener);
        this.setRoomMembership.unregisterListeners();
        this.setRoomState.off('revision', this.stateChangeListener);
        this.setRoomState.unregisterListeners();
        this.watchedPolicyRooms.revisionIssuer.off('revision', this.policyChangeListener);
        this.watchedPolicyRooms.unregisterListeners();
        this.protectedRoomsManager.off('change', this.roomsChangeListener);
        this.protectedRoomsManager.unregisterListeners();
        this.protections.unregisterListeners();
        this.setMembership.off('revision', this.setMembershiprevisionListener);
        this.setMembership.unregisterListeners();
        this.setPoliciesMatchingMembership.off('revision', this.setMembershipPolicyRevisionListener);
        this.setPoliciesMatchingMembership.unregisterListeners();
    }
}
exports.StandardProtectedRoomsSet = StandardProtectedRoomsSet;
//# sourceMappingURL=ProtectedRoomsSet.js.map