"use strict";
// SPDX-FileCopyrightText: 2024 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
Object.defineProperty(exports, "__esModule", { value: true });
const typebox_1 = require("@sinclair/typebox");
const PowerLevelsMirror_1 = require("../Client/PowerLevelsMirror");
const ConfigDescription_1 = require("../Config/ConfigDescription");
const Action_1 = require("../Interface/Action");
const Logger_1 = require("../Logging/Logger");
const MembershipChange_1 = require("../Membership/MembershipChange");
const DeclareRoomState_1 = require("../StateTracking/DeclareRoomState");
const EventGeneration_1 = require("../TestUtilities/EventGeneration");
const log = new Logger_1.Logger('ProtectionHandles.test');
test('handlePermissionRequirementsMet is called when a new room is added with met permissions', async function () {
    const userID = (0, EventGeneration_1.randomUserID)();
    const { protectedRoomsSet, roomStateManager, roomMembershipManager } = await (0, DeclareRoomState_1.describeProtectedRoomsSet)({
        clientUserID: userID,
    });
    const newRoomMatrixID = (0, EventGeneration_1.randomRoomID)([]);
    const newRoom = (0, DeclareRoomState_1.describeRoom)({
        room: newRoomMatrixID,
        membershipDescriptions: [
            {
                sender: userID,
                membership: MembershipChange_1.Membership.Join,
            },
        ],
        stateDescriptions: [
            {
                content: {
                    room_version: '11',
                },
                sender: userID,
                type: 'm.room.create',
                state_key: '',
            },
            {
                content: {
                    users_default: 100,
                    ban: 0,
                },
                type: 'm.room.power_levels',
                sender: userID,
            },
        ],
    });
    let handleCalled = false;
    const protectionDescription = {
        name: 'test',
        description: 'test description',
        capabilities: {},
        defaultCapabilities: {},
        protectionSettings: new ConfigDescription_1.StandardConfigDescription(typebox_1.Type.Object({})),
        async factory(description, _protectedRoomsSet, _context, _capabilities, _settings) {
            return (0, Action_1.Ok)({
                description,
                requiredPermissions: [PowerLevelsMirror_1.PowerLevelPermission.Ban],
                requiredEventPermissions: [],
                requiredStatePermissions: [],
                handlePermissionRequirementsMet(room) {
                    if (room.toRoomIDOrAlias() === newRoomMatrixID.toRoomIDOrAlias()) {
                        handleCalled = true;
                    }
                },
                [Symbol.asyncDispose]() {
                    return Promise.resolve();
                },
            });
        },
    };
    const protectionAddResult = await protectedRoomsSet.protections.addProtection(protectionDescription, protectedRoomsSet, undefined);
    if ((0, Action_1.isError)(protectionAddResult)) {
        throw new TypeError(`Shouldn't fail here`);
    }
    expect(handleCalled).toBe(false);
    roomStateManager.addIssuer(newRoom.stateRevisionIssuer);
    roomMembershipManager.addIssuer(newRoom.membershipRevisionIssuer);
    const addResult = await protectedRoomsSet.protectedRoomsManager.addRoom(newRoom.membershipRevisionIssuer.room);
    if ((0, Action_1.isError)(addResult)) {
        log.error(`Couldn't add room`, addResult);
        throw new TypeError(`Couldn't add a room`);
    }
    expect(handleCalled).toBe(true);
});
//# sourceMappingURL=ProtectionHandles.test.js.map