"use strict";
// Copyright 2023 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MjolnirProtectionsConfig = void 0;
const PersistentConfigData_1 = require("../../Config/PersistentConfigData");
const Action_1 = require("../../Interface/Action");
const Logger_1 = require("../../Logging/Logger");
const Protection_1 = require("../Protection");
const MjolnirEnabledProtectionsDescription_1 = require("./MjolnirEnabledProtectionsDescription");
const log = new Logger_1.Logger('StandardProtectionsConfig');
async function loadProtecitons(config, { migrationHandler, }) {
    const storeResult = await config.requestParsedConfig();
    if ((0, Action_1.isError)(storeResult)) {
        return storeResult;
    }
    const rawData = storeResult.ok;
    if (rawData === undefined && migrationHandler === undefined) {
        return (0, Action_1.Ok)({
            knownEnabledProtections: [],
            unknownEnabledProtections: [],
        });
    }
    const migrateData = async () => {
        const defaultProtections = {
            enabled: [],
        };
        if (migrationHandler === undefined) {
            return (0, Action_1.Ok)(defaultProtections);
        }
        else {
            return await migrationHandler.migrateData(rawData !== null && rawData !== void 0 ? rawData : defaultProtections);
        }
    };
    const migratedData = await migrateData();
    if ((0, Action_1.isError)(migratedData)) {
        log.error(`Unable to migrate raw data `, rawData, migratedData.error);
        return migratedData;
    }
    const knownEnabledProtections = [];
    const unknownEnabledProtections = [];
    for (const protecitonName of migratedData.ok.enabled) {
        const description = (0, Protection_1.findProtection)(protecitonName);
        if (description === undefined) {
            unknownEnabledProtections.push(protecitonName);
        }
        else {
            knownEnabledProtections.push(description);
        }
    }
    return (0, Action_1.Ok)({
        knownEnabledProtections,
        unknownEnabledProtections,
    });
}
function applyMissingProtectionCB(info, cb) {
    if (cb === undefined) {
        return;
    }
    for (const protectionName of info.unknownEnabledProtections) {
        cb(protectionName);
    }
}
async function storeProtections(info, config, { enabledProtectionsMigration, }) {
    const combinedEnabledProtections = new Set([
        ...info.knownEnabledProtections.map((protection) => protection.name),
        ...info.unknownEnabledProtections,
    ]);
    return await config.saveConfig({
        enabled: [...combinedEnabledProtections],
        ...(enabledProtectionsMigration === undefined
            ? {}
            : {
                [enabledProtectionsMigration.versionKey]: enabledProtectionsMigration.latestVersion,
            }),
    });
}
class MjolnirProtectionsConfig {
    constructor(config, logTracker, info, migrationHandler) {
        this.config = config;
        this.info = info;
        this.migrationHandler = migrationHandler;
        logTracker.addLoggableConfig(this);
    }
    static async create(store, logTracker, { migrationHandler, 
    /**
     * It is necessary for some consumers to provide a way to enable/disable protections
     * based the version of software that is being loaded. For example Draupnir
     * needs to enable the `BanPropagationProtection` for users who are upgrading
     * from older versions & for those migrating from Mjolnir.
     * This should not be used to change the structure of the account data itself,
     * because this is supposed to be directly compatible with Mjolnir account data.
     */
    missingProtectionCB, }) {
        const config = new PersistentConfigData_1.StandardPersistentConfigData(MjolnirEnabledProtectionsDescription_1.MjolnirEnabledProtectionsDescription, store);
        const protectionsInfo = await loadProtecitons(config, { migrationHandler });
        if ((0, Action_1.isError)(protectionsInfo)) {
            return protectionsInfo;
        }
        applyMissingProtectionCB(protectionsInfo.ok, missingProtectionCB);
        return (0, Action_1.Ok)(new MjolnirProtectionsConfig(config, logTracker, protectionsInfo.ok, migrationHandler));
    }
    logCurrentConfig() {
        log.info(`current config:`, this.info);
    }
    async enableProtection(protectionDescription) {
        const nextInfo = {
            knownEnabledProtections: [
                protectionDescription,
                ...this.getKnownEnabledProtections(),
            ],
            unknownEnabledProtections: this.getUnknownEnabledProtections(),
        };
        const storeResult = await storeProtections(nextInfo, this.config, {
            enabledProtectionsMigration: this.migrationHandler,
        });
        if ((0, Action_1.isOk)(storeResult)) {
            this.info = nextInfo;
        }
        return storeResult;
    }
    async disableProtection(protectionName) {
        const nextInfo = {
            knownEnabledProtections: this.getKnownEnabledProtections().filter((description) => description.name !== protectionName),
            unknownEnabledProtections: this.getUnknownEnabledProtections().filter((name) => name !== protectionName),
        };
        const storeResult = await storeProtections(nextInfo, this.config, {
            enabledProtectionsMigration: this.migrationHandler,
        });
        if ((0, Action_1.isOk)(storeResult)) {
            this.info = nextInfo;
        }
        return storeResult;
    }
    getKnownEnabledProtections() {
        return this.info.knownEnabledProtections;
    }
    getUnknownEnabledProtections() {
        return this.info.unknownEnabledProtections;
    }
}
exports.MjolnirProtectionsConfig = MjolnirProtectionsConfig;
//# sourceMappingURL=StandardProtectionsConfig.js.map