"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-protection-suite
// https://github.com/Gnuxie/matrix-protection-suite
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardServerBanIntentProjectionNode = exports.ServerBanIntentProjectionHelper = void 0;
const PolicyRuleChange_1 = require("../../../PolicyList/PolicyRuleChange");
const PolicyRule_1 = require("../../../PolicyList/PolicyRule");
const immutable_1 = require("immutable");
const PolicyEvents_1 = require("../../../MatrixTypes/PolicyEvents");
const ListMultiMap_1 = require("../../../Projection/ListMultiMap");
exports.ServerBanIntentProjectionHelper = Object.freeze({
    reducePolicyDelta(input) {
        const output = {
            add: [],
            remove: [],
        };
        for (const change of input) {
            if (change.rule.kind !== PolicyEvents_1.PolicyRuleType.Server) {
                continue;
            }
            else if (change.rule.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                continue;
            }
            switch (change.changeType) {
                case PolicyRuleChange_1.PolicyRuleChangeType.Added:
                case PolicyRuleChange_1.PolicyRuleChangeType.RevealedLiteral: {
                    output.add.push(change.rule);
                    break;
                }
                case PolicyRuleChange_1.PolicyRuleChangeType.Modified: {
                    output.add.push(change.rule);
                    if (change.previousRule === undefined) {
                        throw new TypeError('Things are very wrong');
                    }
                    output.remove.push(change.previousRule);
                    break;
                }
                case PolicyRuleChange_1.PolicyRuleChangeType.Removed: {
                    output.remove.push(change.rule);
                    break;
                }
            }
        }
        return output;
    },
    reduceIntentDelta(input, policies) {
        const intents = ListMultiMap_1.ListMultiMap.deriveIntents(policies, input.add, input.remove, (rule) => rule.entity);
        return {
            ...input,
            deny: intents.intend,
            recall: intents.recall,
        };
    },
});
class StandardServerBanIntentProjectionNode {
    constructor(ulidFactory, policies) {
        this.ulidFactory = ulidFactory;
        this.policies = policies;
        this.ulid = ulidFactory();
    }
    static create(ulidFactory) {
        return new StandardServerBanIntentProjectionNode(ulidFactory, (0, immutable_1.Map)());
    }
    reduceInput(input) {
        return exports.ServerBanIntentProjectionHelper.reduceIntentDelta(exports.ServerBanIntentProjectionHelper.reducePolicyDelta(input), this.policies);
    }
    reduceInitialInputs([policyListRevision]) {
        if (!this.isEmpty()) {
            throw new TypeError('Cannot reduce initial inputs when inialised');
        }
        const serverPolicies = [
            ...policyListRevision.allRulesOfType(PolicyEvents_1.PolicyRuleType.Server, PolicyRule_1.Recommendation.Ban),
            ...policyListRevision.allRulesOfType(PolicyEvents_1.PolicyRuleType.Server, PolicyRule_1.Recommendation.Takedown),
        ].filter((rule) => rule.matchType !== PolicyRule_1.PolicyRuleMatchType.HashedLiteral);
        const names = new Set(serverPolicies.map((policy) => policy.entity));
        return {
            add: serverPolicies,
            deny: [...names],
            remove: [],
            recall: [],
        };
    }
    isEmpty() {
        return this.policies.size === 0;
    }
    reduceDelta(input) {
        let nextPolicies = this.policies;
        nextPolicies = ListMultiMap_1.ListMultiMap.addValues(nextPolicies, input.add, (rule) => rule.entity);
        nextPolicies = ListMultiMap_1.ListMultiMap.removeValues(nextPolicies, input.remove, (rule) => rule.entity);
        return new StandardServerBanIntentProjectionNode(this.ulidFactory, nextPolicies);
    }
    get deny() {
        return [...this.policies.keys()];
    }
}
exports.StandardServerBanIntentProjectionNode = StandardServerBanIntentProjectionNode;
//# sourceMappingURL=ServerBanIntentProjectionNode.js.map