"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 2022 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerBanSynchronisationProtection = void 0;
const Action_1 = require("../../../Interface/Action");
const Task_1 = require("../../../Interface/Task");
const Protection_1 = require("../../Protection");
require("./ServerBanSynchronisationCapability");
require("./ServerACLSynchronisationCapability");
const ServerBanIntentProjection_1 = require("./ServerBanIntentProjection");
const Logger_1 = require("../../../Logging/Logger");
const log = new Logger_1.Logger('ServerBanSynchronisationProtection');
// FIXME: We need a linear gate around the server ACL consequence for the entire
// room set.
class ServerBanSynchronisationProtection extends Protection_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, intentProjection) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.intentProjection = intentProjection;
        this.capability = capabilities.serverConsequences;
    }
    // TODO: We really need a loop detection thing here, we can borrow the infringement
    // count utility from draupinr protections to see if we can unprotect the room
    // if this handle keeps being effectual.
    async handleStateChange(revision, changes) {
        const serverACLEventChanges = changes.filter((change) => change.eventType === 'm.room.server_acl');
        if (serverACLEventChanges.length === 0) {
            return (0, Action_1.Ok)(undefined);
        }
        if (serverACLEventChanges.length !== 1) {
            throw new TypeError(`How is it possible for there to be more than one server_acl event change in the same revision?`);
        }
        return (await this.capability.outcomeFromIntentInRoom(revision.room.toRoomIDOrAlias(), this.intentProjection));
    }
    handleIntentProjectionNode() {
        void (0, Task_1.Task)(this.capability.outcomeFromIntentInRoomSet(this.intentProjection), { log });
    }
    handlePermissionRequirementsMet(room) {
        void (0, Task_1.Task)((async () => {
            await this.capability.outcomeFromIntentInRoom(room.toRoomIDOrAlias(), this.intentProjection);
        })());
    }
}
exports.ServerBanSynchronisationProtection = ServerBanSynchronisationProtection;
(0, Protection_1.describeProtection)({
    name: 'ServerBanSynchronisationProtection',
    description: 'Synchronise server bans from watched policy lists across the protected rooms set by producing ServerACL events',
    capabilityInterfaces: {
        serverConsequences: 'ServerBanSynchronisationCapability',
    },
    defaultCapabilities: {
        serverConsequences: 'ServerACLSynchronisationCapability',
    },
    factory: async (description, lifetime, protectedRoomsSet, _settings, capabilities) => {
        const intentProjection = lifetime.allocateDisposable(() => (0, Action_1.Ok)(new ServerBanIntentProjection_1.StandardServerBanIntentProjection(protectedRoomsSet.watchedPolicyRooms.revisionIssuer)));
        if ((0, Action_1.isError)(intentProjection)) {
            return intentProjection.elaborate('Unable to allocate intent projection');
        }
        return (0, Action_1.Ok)(new ServerBanSynchronisationProtection(description, lifetime, capabilities, protectedRoomsSet, intentProjection.ok));
    },
});
//# sourceMappingURL=ServerBanSynchronisation.js.map