"use strict";
// SPDX-FileCopyrightText: 2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardSHA256HashReverser = void 0;
const PolicyRule_1 = require("../../../PolicyList/PolicyRule");
const PolicyRuleChange_1 = require("../../../PolicyList/PolicyRuleChange");
const events_1 = __importDefault(require("events"));
const typescript_result_1 = require("@gnuxie/typescript-result");
const PolicyEvents_1 = require("../../../MatrixTypes/PolicyEvents");
const Logger_1 = require("../../../Logging/Logger");
const Task_1 = require("../../../Interface/Task");
const Reversal_1 = require("./Reversal");
const DirectPropagationPolicyListRevisionIssuer_1 = require("../../DirectPropagationPolicyListRevisionIssuer");
const log = new Logger_1.Logger('SHA256HashReverser');
class StandardSHA256HashReverser extends events_1.default {
    constructor(store) {
        super();
        this.store = store;
        this.hashedPoliciesRevisionIssuer = new DirectPropagationPolicyListRevisionIssuer_1.StandardDirectPropagationPolicyListRevisionIssuer([], function (change) {
            if (change.rule.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                return true;
                // We do this because we need to allow reversed policies through to remove hashed literals that
                // have been revealed, removed, or modified.
            }
            else if (change.rule.isReversedFromHashedPolicy) {
                return true;
            }
            else {
                return false;
            }
        });
        this.issuers = new Map();
        this.handleDiscoveredHashes = function (roomHashes, userHashes, serverHashes) {
            const reversedPolicies = [
                ...(0, Reversal_1.reversePoliciesOfType)(roomHashes, (record) => record.room_id, PolicyEvents_1.PolicyRuleType.Room, this.hashedPoliciesRevisionIssuer.currentRevision),
                ...(0, Reversal_1.reversePoliciesOfType)(userHashes, (record) => record.user_id, PolicyEvents_1.PolicyRuleType.User, this.hashedPoliciesRevisionIssuer.currentRevision),
                ...(0, Reversal_1.reversePoliciesOfType)(serverHashes, (record) => record.server_name, PolicyEvents_1.PolicyRuleType.Server, this.hashedPoliciesRevisionIssuer.currentRevision),
            ];
            this.updateUpstreamWithRevealedPolicies(reversedPolicies);
        }.bind(this);
        this.handlePolicyRoomRevision = function (revision, mixedChanges) {
            void (0, Task_1.Task)((async () => {
                const addedPoliciesToCheck = [];
                for (const change of mixedChanges) {
                    if (change.changeType === PolicyRuleChange_1.PolicyRuleChangeType.Added &&
                        change.rule.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral) {
                        addedPoliciesToCheck.push(change.rule);
                    }
                }
                await this.checkPoliciesAgainstStore(addedPoliciesToCheck);
            })());
        }.bind(this);
        this.store.on('ReversedHashes', this.handleDiscoveredHashes);
        this.hashedPoliciesRevisionIssuer.on('revision', this.handlePolicyRoomRevision);
    }
    addPolicyRoomRevisionIssuer(issuer) {
        this.hashedPoliciesRevisionIssuer.addIssuer(issuer);
        this.issuers.set(issuer.room.toRoomIDOrAlias(), issuer);
        void (0, Task_1.Task)(this.checkPoliciesAgainstStore(issuer.currentRevision
            .allRules()
            .filter((policy) => policy.matchType === PolicyRule_1.PolicyRuleMatchType.HashedLiteral)));
    }
    updateUpstreamWithRevealedPolicies(reversedPolicies) {
        const policiesByPolicyRoom = new Map();
        for (const policy of reversedPolicies) {
            const entry = policiesByPolicyRoom.get(policy.sourceEvent.room_id);
            if (entry === undefined) {
                policiesByPolicyRoom.set(policy.sourceEvent.room_id, [policy]);
            }
            else {
                entry.push(policy);
            }
        }
        for (const [roomID, policies] of policiesByPolicyRoom) {
            const issuer = this.issuers.get(roomID);
            if (issuer === undefined) {
                throw new TypeError('Somehow this revision issuer is out of sync');
            }
            issuer.updateForRevealedPolicies(policies);
        }
    }
    async checkPoliciesAgainstStore(policies) {
        const newlyReversedPolicies = await this.store.reverseHashedPolicies(policies);
        if ((0, typescript_result_1.isError)(newlyReversedPolicies)) {
            log.error('Unable to reverse new policies', newlyReversedPolicies.error);
            return;
        }
        this.updateUpstreamWithRevealedPolicies(newlyReversedPolicies.ok);
    }
    unregisterListeners() {
        this.store.off('ReversedHashes', this.handleDiscoveredHashes);
        this.hashedPoliciesRevisionIssuer.off('revision', this.handlePolicyRoomRevision);
        this.hashedPoliciesRevisionIssuer.unregisterListeners();
    }
}
exports.StandardSHA256HashReverser = StandardSHA256HashReverser;
//# sourceMappingURL=SHA256HashReverser.js.map