"use strict";
// Copyright 2022 - 2025 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from matrix-protection-suite
// https://github.com/Gnuxie/matrix-protection-suite
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConstantPeriodEventBatch = exports.ConstantPeriodItemBatch = exports.StandardBatcher = void 0;
const Logger_1 = require("../Logging/Logger");
const log = new Logger_1.Logger('EventBatch');
function logBatchCompleteCallbackError(e) {
    log.error('Caught an exception from the callback for an event batch', e);
}
class StandardBatcher {
    constructor(batchFactoryMethod) {
        this.batchFactoryMethod = batchFactoryMethod;
        this.currentBatch = batchFactoryMethod();
    }
    add(key, value) {
        if (this.currentBatch.isFinished()) {
            this.currentBatch = this.batchFactoryMethod();
        }
        this.currentBatch.add(key, value);
    }
    dispose() {
        if (!this.currentBatch.isFinished()) {
            this.currentBatch.cancel();
        }
    }
}
exports.StandardBatcher = StandardBatcher;
class ConstantPeriodItemBatch {
    constructor(batchCompleteCallback, { waitPeriodMS = 200 }) {
        this.batchCompleteCallback = batchCompleteCallback;
        this.items = new Map();
        this.isBatchComplete = false;
        this.timeoutID = undefined;
        this.waitPeriodMS = waitPeriodMS;
    }
    isFinished() {
        return this.isBatchComplete;
    }
    add(key, item) {
        if (this.isFinished()) {
            throw new TypeError('Something tried adding an event to a completed EventBatch');
        }
        if (this.items.has(key)) {
            return;
        }
        this.items.set(key, item);
        if (!this.timeoutID) {
            // spawn off the timer to call the callback.
            this.startCallbackTimer();
        }
    }
    startCallbackTimer() {
        if (this.timeoutID) {
            throw new TypeError('The callback timer is being started more than once');
        }
        this.timeoutID = setTimeout(this.completeBatch.bind(this), this.waitPeriodMS);
    }
    completeBatch() {
        this.isBatchComplete = true;
        this.batchCompleteCallback([...this.items.entries()]).catch(logBatchCompleteCallbackError);
    }
    cancel() {
        if (this.timeoutID) {
            clearTimeout(this.timeoutID);
            this.timeoutID = undefined;
        }
    }
}
exports.ConstantPeriodItemBatch = ConstantPeriodItemBatch;
class ConstantPeriodEventBatch {
    constructor(batchCompleteCallback, { waitPeriodMS = 200 }) {
        this.batchCompleteCallback = batchCompleteCallback;
        this.batch = new ConstantPeriodItemBatch((entries) => this.batchCompleteCallback(entries.map((entry) => entry[1])), { waitPeriodMS });
    }
    isFinished() {
        return this.batch.isFinished();
    }
    addEvent(event) {
        this.batch.add(event.event_id, event);
    }
}
exports.ConstantPeriodEventBatch = ConstantPeriodEventBatch;
//# sourceMappingURL=EventBatch.js.map