"use strict";
// Copyright (C) 2023-2025 Gnuxie <Gnuxie@protonmail.com>
//
// SPDX-License-Identifier: AFL-3.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandardRoomStateRevisionIssuer = void 0;
const events_1 = __importDefault(require("events"));
const StandardRoomStateRevision_1 = require("./StandardRoomStateRevision");
const EventBatch_1 = require("./EventBatch");
const Action_1 = require("../Interface/Action");
const Logger_1 = require("../Logging/Logger");
const Redaction_1 = require("../MatrixTypes/Redaction");
const StateChangeType_1 = require("./StateChangeType");
const await_lock_1 = __importDefault(require("await-lock"));
const log = new Logger_1.Logger('StandardRoomStateRevisionIssuer');
class StandardRoomStateRevisionIssuer extends events_1.default {
    constructor(room, roomStateGetter, initialState) {
        super();
        this.room = room;
        this.roomStateGetter = roomStateGetter;
        this.stateRefreshLock = new await_lock_1.default();
        this.currentRevision = StandardRoomStateRevision_1.StandardRoomStateRevision.blankRevision(this.room).reviseFromState(initialState);
        this.batchCompleteCallback = this.createBatchedRevision.bind(this);
        this.currentBatch = new EventBatch_1.ConstantPeriodEventBatch(this.batchCompleteCallback, {});
    }
    addEventToBatch(event) {
        if (this.currentBatch.isFinished()) {
            this.currentBatch = new EventBatch_1.ConstantPeriodEventBatch(this.batchCompleteCallback, {});
        }
        this.currentBatch.addEvent(event);
    }
    updateForEvent(event) {
        if (this.currentRevision.hasEvent(event.event_id)) {
            return;
        }
        const existingState = this.currentRevision.getStateEvent(event.type, event.state_key);
        if (existingState === undefined) {
            this.createRevisionFromChanges([
                {
                    changeType: (0, StateChangeType_1.calculateStateChange)(event, existingState),
                    eventType: event.type,
                    state: event,
                },
            ]);
        }
        else {
            // state already exists for the type+key combo
            // we need to ask the homeserver to determine how state has changed
            this.addEventToBatch(event);
        }
    }
    updateForRedaction(event) {
        const targetEvent = (0, Redaction_1.redactionTargetEvent)(event);
        if (targetEvent === undefined) {
            log.warn(`Someone has been redacting redaction events, interesting`, targetEvent);
            return;
        }
        if (!this.currentRevision.hasEvent(targetEvent)) {
            return;
        }
        this.addEventToBatch(event);
    }
    createRevisionFromChanges(changes) {
        const previousRevision = this.currentRevision;
        this.currentRevision = this.currentRevision.reviseFromChanges(changes);
        this.emit('revision', this.currentRevision, changes, previousRevision);
    }
    async createBatchedRevision() {
        await this.stateRefreshLock.acquireAsync();
        try {
            const currentRoomStateResult = await this.roomStateGetter.getAllState(this.room);
            if ((0, Action_1.isError)(currentRoomStateResult)) {
                log.error(`Unable to fetch state from the room ${this.room.toPermalink()}.`, currentRoomStateResult.error);
                return;
            }
            const changes = this.currentRevision.changesFromState(currentRoomStateResult.ok);
            this.createRevisionFromChanges(changes);
        }
        finally {
            this.stateRefreshLock.release();
        }
    }
    async refreshRoomState() {
        await this.createBatchedRevision();
    }
    unregisterListeners() {
        // nothing to do.
    }
}
exports.StandardRoomStateRevisionIssuer = StandardRoomStateRevisionIssuer;
//# sourceMappingURL=StandardRoomStateRevisionIssuer.js.map