"use strict";
// Copyright 2022 - 2024 Gnuxie <Gnuxie@protonmail.com>
// Copyright 2019 - 2021 The Matrix.org Foundation C.I.C.
//
// SPDX-License-Identifier: AFL-3.0 AND Apache-2.0
//
// SPDX-FileAttributionText: <text>
// This modified file incorporates work from mjolnir
// https://github.com/matrix-org/mjolnir
// </text>
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessageIsMediaProtection = void 0;
const matrix_bot_sdk_1 = require("matrix-bot-sdk");
const matrix_protection_suite_1 = require("matrix-protection-suite");
const matrix_basic_types_1 = require("@the-draupnir-project/matrix-basic-types");
(0, matrix_protection_suite_1.describeProtection)({
    name: "MessageIsMediaProtection",
    description: "If a user posts an image or video, that message will be redacted. No bans are issued.",
    capabilityInterfaces: {
        eventConsequences: "EventConsequences",
    },
    defaultCapabilities: {
        eventConsequences: "StandardEventConsequences",
    },
    factory: async function (description, lifetime, protectedRoomsSet, draupnir, capabilities, _settings) {
        return (0, matrix_protection_suite_1.Ok)(new MessageIsMediaProtection(description, lifetime, capabilities, protectedRoomsSet, draupnir));
    },
});
class MessageIsMediaProtection extends matrix_protection_suite_1.AbstractProtection {
    constructor(description, lifetime, capabilities, protectedRoomsSet, draupnir) {
        super(description, lifetime, capabilities, protectedRoomsSet, {});
        this.draupnir = draupnir;
        this.eventConsequences = capabilities.eventConsequences;
    }
    async handleTimelineEvent(room, event) {
        if (matrix_protection_suite_1.Value.Check(matrix_protection_suite_1.RoomMessage, event)) {
            if (!("msgtype" in event.content)) {
                return (0, matrix_protection_suite_1.Ok)(undefined);
            }
            const msgtype = event.content["msgtype"];
            const formattedBody = "formatted_body" in event.content
                ? event.content["formatted_body"] || ""
                : "";
            const isMedia = msgtype === "m.image" ||
                msgtype === "m.video" ||
                formattedBody.toLowerCase().includes("<img");
            if (isMedia) {
                const roomID = room.toRoomIDOrAlias();
                await this.draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "MessageIsMedia", `Redacting event from ${event["sender"]} for posting an image/video. ${matrix_basic_types_1.Permalinks.forEvent(roomID, event["event_id"], [(0, matrix_basic_types_1.userServerName)(this.draupnir.clientUserID)])}`);
                // Redact the event
                if (!this.draupnir.config.noop) {
                    await this.eventConsequences.consequenceForEvent(roomID, event["event_id"], "Images/videos are not permitted here");
                }
                else {
                    await this.draupnir.managementRoomOutput.logMessage(matrix_bot_sdk_1.LogLevel.WARN, "MessageIsMedia", `Tried to redact ${event["event_id"]} in ${roomID} but Draupnir is running in no-op mode`, roomID);
                }
            }
        }
        return (0, matrix_protection_suite_1.Ok)(undefined);
    }
}
exports.MessageIsMediaProtection = MessageIsMediaProtection;
//# sourceMappingURL=MessageIsMedia.js.map