/*
 * Copyright (c) 2020 EKA2L1 Team.
 * 
 * This file is part of EKA2L1 project 
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <common/buffer.h>
#include <common/chunkyseri.h>
#include <utils/cardinality.h>

namespace eka2l1::utils {
    bool cardinality::internalize(common::ro_stream &stream) {
        std::uint8_t b1 = 0;
        std::uint32_t len = 0;

        if (stream.read(&b1, 1) != 1) {
            return false;
        }

        if ((b1 & 1) == 0) {
            len = (b1 >> 1);
        } else {
            if ((b1 & 2) == 0) {
                len = b1;

                if (stream.read(&b1, 1) != 1) {
                    return false;
                }

                len += b1 << 8;
                len >>= 2;
            } else if ((b1 & 4) == 0) {
                std::uint16_t b2 = 0;

                if (stream.read(&b2, 2) != 2) {
                    return false;
                }

                len = b1 + (b2 << 8);
                len >>= 4;
            }
        }

        val_ = len;
        return true;
    }

    bool cardinality::externalize(common::wo_stream &stream) {
        if (val_ <= (0xFF >> 1)) {
            std::uint8_t b = static_cast<std::uint8_t>(val_ << 1);
            return (stream.write(&b, 1) == 1);
        }

        if (val_ <= (0xFFFF >> 2)) {
            std::uint16_t b = static_cast<std::uint16_t>(val_ << 2) + 0x1;
            return (stream.write(&b, 2) == 2);
        }

        std::uint32_t b = (val_ << 3) + 0x3;
        return (stream.write(&b, 4) == 4);
    }
    
    void cardinality::serialize(common::chunkyseri &seri) {
        if (seri.get_seri_mode() != common::SERI_MODE_READ) {
            if (val_ <= (0xFF >> 1)) {
                std::uint8_t b = static_cast<std::uint8_t>(val_ << 1);
                seri.absorb(b);
            } else {
                if (val_ <= (0xFFFF >> 2)) {
                    std::uint16_t b = static_cast<std::uint16_t>(val_ << 2) + 0x1;
                    seri.absorb(b);
                } else {
                    std::uint32_t b = (val_ << 3) + 0x3;
                    seri.absorb(b);
                }
            }
        } else {
            std::uint8_t b1 = 0;
            seri.absorb(b1);

            if ((b1 & 1) == 0) {
                val_ = (b1 >> 1);
            } else {
                if ((b1 & 2) == 0) {
                    val_ = b1;
                    seri.absorb(b1);
                    val_ += b1 << 8;
                    val_ >>= 2;
                } else if ((b1 & 4) == 0) {
                    std::uint16_t b2 = 0;
                    seri.absorb(b2);
                    val_ = b1 + (b2 << 8);
                    val_ >>= 4;
                }
            }
        }
    }
}