/*
 * %W% %E%
 *
 * Copyright (c) 1996 Sun Microsystems, Inc. All Rights Reserved.
 *
 * This software is the confidential and proprietary information of Sun
 * Microsystems, Inc. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Sun.
 *
 * SUN MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
 * SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. SUN SHALL NOT BE LIABLE FOR ANY DAMAGES
 * SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR DISTRIBUTING
 * THIS SOFTWARE OR ITS DERIVATIVES.
 *
 */

/*
 * g721.c
 *
 * Description:
 *
 * g721_encoder(), g721_decoder()
 *
 * These routines comprise an implementation of the CCITT G.721 ADPCM
 * coding algorithm.  Essentially, this implementation is identical to
 * the bit level description except for a few deviations which
 * take advantage of work station attributes, such as hardware 2's
 * complement arithmetic and large memory.  Specifically, certain time
 * consuming operations such as multiplications are replaced
 * with lookup tables and software 2's complement operations are
 * replaced with hardware 2's complement.
 *
 * The deviation from the bit level specification (lookup tables)
 * preserves the bit level performance specifications.
 *
 * As outlined in the G.721 Recommendation, the algorithm is broken
 * down into modules.  Each section of code below is preceded by
 * the name of the module which it is implementing.
 *
 */
#include "g72x.h"

#if USE_HIGHLEVEL_FILE_API != FALSE

static short qtab_721[7] = {-124, 80, 178, 246, 300, 349, 400};
/*
 * Maps G.721 code word to reconstructed scale factor normalized log
 * magnitude values.
 */
static const short  _dqlntab[16] = {-2048, 4, 135, 213, 273, 323, 373, 425,
                            425, 373, 323, 273, 213, 135, 4, -2048};

/* Maps G.721 code word to log of scale factor multiplier. */
static const short  _witab[16] = {-12, 18, 41, 64, 112, 198, 355, 1122,
                            1122, 355, 198, 112, 64, 41, 18, -12};
/*
 * Maps G.721 code words to a set of values whose long and short
 * term averages are computed and then compared to give an indication
 * how stationary (steady state) the signal is.
 */
static const short  _fitab[16] = {0, 0, 0, 0x200, 0x200, 0x200, 0x600, 0xE00,
                            0xE00, 0x600, 0x200, 0x200, 0x200, 0, 0, 0};

#if 0
/*
 * g721_encoder()
 *
 * Encodes the input vale of linear PCM, A-law or u-law data sl and returns
 * the resulting code. -1 is returned for unknown input coding value.
 */
int
g721_encoder(
    int     sl,
    int     in_coding,
    struct g72x_state *state_ptr)
{
    short       sezi, se, sez;      /* ACCUM */
    short       d;          /* SUBTA */
    short       sr;         /* ADDB */
    short       y;          /* MIX */
    short       dqsez;          /* ADDC */
    short       dq, i;

    switch (in_coding) {    /* linearize input sample to 14-bit PCM */
    case AUDIO_ENCODING_ALAW:
        sl = alaw2linear(sl) >> 2;
        break;
    case AUDIO_ENCODING_ULAW:
        sl = ulaw2linear(sl) >> 2;
        break;
    case AUDIO_ENCODING_LINEAR:
        sl >>= 2;           /* 14-bit dynamic range */
        break;
    default:
        return (-1);
    }

    sezi = predictor_zero(state_ptr);
    sez = sezi >> 1;
    se = (sezi + predictor_pole(state_ptr)) >> 1;   /* estimated signal */

    d = sl - se;                /* estimation difference */

    /* quantize the prediction difference */
    y = step_size(state_ptr);       /* quantizer step size */
    i = quantize(d, y, qtab_721, 7);    /* i = ADPCM code */

    dq = reconstruct(i & 8, _dqlntab[i], y);    /* quantized est diff */

    sr = (dq < 0) ? se - (dq & 0x3FFF) : se + dq;   /* reconst. signal */

    dqsez = sr + sez - se;          /* pole prediction diff. */

    update(4, y, _witab[i] << 5, _fitab[i], dq, sr, dqsez, state_ptr);

    return (i);
}
#endif

/*
 * g721_decoder()
 *
 * Description:
 *
 * Decodes a 4-bit code of G.721 encoded data of i and
 * returns the resulting linear PCM, A-law or u-law value.
 * return -1 for unknown out_coding value.
 */
int
g721_decoder(
    int     i,
    int     out_coding,
    struct g72x_state *state_ptr)
{
    short       sezi, sei, sez, se; /* ACCUM */
    short       y;          /* MIX */
    short       sr;         /* ADDB */
    short       dq;
    short       dqsez;

    i &= 0x0f;          /* mask to get proper bits */
    sezi = predictor_zero(state_ptr);
    sez = sezi >> 1;
    sei = sezi + predictor_pole(state_ptr);
    se = sei >> 1;          /* se = estimated signal */

    y = step_size(state_ptr);   /* dynamic quantizer step size */

    dq = reconstruct(i & 0x08, _dqlntab[i], y); /* quantized diff. */

    sr = (dq < 0) ? (se - (dq & 0x3FFF)) : se + dq; /* reconst. signal */

    dqsez = sr - se + sez;          /* pole prediction diff. */

    update(4, y, _witab[i] << 5, _fitab[i], dq, sr, dqsez, state_ptr);

    switch (out_coding) {
    case AUDIO_ENCODING_ALAW:
        return (tandem_adjust_alaw(sr, se, y, i, 8, qtab_721));
    case AUDIO_ENCODING_ULAW:
        return (tandem_adjust_ulaw(sr, se, y, i, 8, qtab_721));
    case AUDIO_ENCODING_LINEAR:
        return (sr << 2);   /* sr was 14-bit dynamic range */
    default:
        return (-1);
    }
}
#endif
