# 11 "src/lexer.mll"
 

open Lexing
open Parser
open Positions
open Keyword
open Attribute
open Syntax

(* ------------------------------------------------------------------------ *)

(* Short-hands. *)

let error1 pos =
  Error.error (Positions.one pos)

let error2 lexbuf =
  Error.error (Positions.lexbuf lexbuf)

(* ------------------------------------------------------------------------ *)

(* [int_of_string] raises [Failure] if its argument is too large. This is
   not a problem in practice, but causes false positives when fuzzing
   Menhir. We hide the problem by failing gracefully. *)

let int_of_string (pos : Lexing.position) i =
  try
    int_of_string i
  with Failure _ ->
    error1 pos "unreasonably large integer."

(* ------------------------------------------------------------------------ *)

(* This wrapper saves the current lexeme start, invokes its argument,
   and restores it. This allows transmitting better positions to the
   parser. *)

let savestart lexbuf f =
  let startp = lexbuf.lex_start_p in
  let token = f lexbuf in
  lexbuf.lex_start_p <- startp;
  token

(* ------------------------------------------------------------------------ *)

(* Overwrites an old character with a new one at a specified
   offset in a [bytes] buffer. *)

let overwrite content offset c1 c2 =
  assert (Bytes.get content offset = c1);
  Bytes.set content offset c2

(* ------------------------------------------------------------------------ *)

(* Keyword recognition and construction. *)

(* A monster is a spot where we have identified a keyword in concrete syntax.
   We describe a monster as an object with the following methods: *)

type monster = {

  (* The position of the monster. *)
  pos: Positions.t;

  (* This method is passed an array of (optional) names for the producers,
     that is, the elements of the production's right-hand side. It is also
     passed a flag which tells whether [$i] syntax is allowed or disallowed.
     It may perform some checks and is allowed to fail. *)
  check: check;

  (* This method transforms the keyword (in place) into a conventional
     OCaml identifier. This is done by replacing '$', '(', and ')' with
     '_'. Bloody. The arguments are [ofs1] and [content]. [ofs1] is the
     offset where [content] begins in the source file. *)
  transform: int -> bytes -> unit;

  (* This is the keyword, in abstract syntax. *)
  keyword: keyword option;

  (* If this is a [$i] monster, then the identifier [_i] is stored here. *)
  oid: string option;

}

(* The reason why we parameterize [check] over [Settings.dollars], instead
   of reading this setting directly, is that [NewRuleSyntax] calls [check]
   and wants to disallow dollars. *)
and check =
  Settings.dollars -> string option array -> unit

(* No check. *)

let none : check =
  fun _ _ -> ()

(* ------------------------------------------------------------------------ *)

(* We check that every [$i] is within range. Also, we forbid using [$i]
   when a producer has been given a name; this is bad style and may be
   a mistake. (Plus, this simplifies our life, as we rewrite [$i] to [_i],
   and we would have to rewrite it to a different identifier otherwise.) *)

let check_dollar pos i : check = fun dollars producers ->
  (* If [i] is out of range, say so. *)
  if not (0 <= i - 1 && i - 1 < Array.length producers) then
    Error.error [pos] "$%d refers to a nonexistent symbol." i;
  (* If [$i] could be referred to via a name, say so. *)
  producers.(i - 1) |> Option.iter (fun x ->
    Error.error [pos] "please do not say: $%d. Instead, say: %s." i x
  );
  (* If [$i] syntax is disallowed, say so. *)
  match dollars with
  | Settings.DollarsDisallowed ->
      Error.error [pos] "please do not use $%d. Instead, name this value." i
  | Settings.DollarsAllowed ->
      ()

(* We check that every reference to a producer [x] in a position keyword,
   such as [$startpos(x)], exists. *)

let check_producer pos x : check = fun _ producers ->
  if not (List.mem (Some x) (Array.to_list producers)) then
    Error.error [pos] "%s refers to a nonexistent symbol." x

(* ------------------------------------------------------------------------ *)

(* The [$i] monster. *)

let dollar pos i : monster =
  let check : check = check_dollar pos i
  and transform ofs1 content =
    (* [$i] is replaced with [_i]. Thus, it is no longer a keyword. *)
    let pos = start_of_position pos in
    let ofs = pos.pos_cnum - ofs1 in
    overwrite content ofs '$' '_'
  and keyword =
    None
  and oid =
    Some (Printf.sprintf "_%d" i)
  in
  { pos; check; transform; keyword; oid }

(* ------------------------------------------------------------------------ *)

(* The position-keyword monster. The most horrible of all. *)

let position pos
  (where : string)
  (flavor : string)
  (i : string option) (x : string option)
=
  let check_no_parameter () =
    if i <> None || x <> None then
      Error.error [pos] "$%s%s does not take a parameter." where flavor
  in
  let ofslpar = (* offset of the opening parenthesis, if there is one *)
    1 + (* for the initial "$" *)
    String.length where +
    3   (* for "pos" or "ofs" or "loc" *)
  in
  let where =
    match where with
    | "symbolstart"
    | "s"           -> check_no_parameter(); WhereSymbolStart
    | "start"       -> WhereStart
    | "end"         -> WhereEnd
    | ""            -> WhereStart
    | _             -> assert false
  in
  let flavor =
    match flavor with
    | "pos"   -> FlavorPosition
    | "ofs"   -> FlavorOffset
    | "loc"   -> FlavorLocation
    | _       -> assert false
  in
  let subject, check =
    match i, x with
    | Some i, None ->
        let ii = int_of_string (start_of_position pos) i in
        if ii = 0 && where = WhereEnd then
          (* [$endpos($0)] *)
          Before, none
        else
          (* [$startpos($i)] is rewritten to [$startpos(_i)]. *)
          RightNamed ("_" ^ i), check_dollar pos ii
    | None, Some x ->
        (* [$startpos(x)] *)
        RightNamed x, check_producer pos x
    | None, None ->
        (* [$startpos] *)
        Left, none
    | Some _, Some _ ->
        assert false
  in
  let transform ofs1 content =
    let pos = start_of_position pos in
    let ofs = pos.pos_cnum - ofs1 in
    overwrite content ofs '$' '_';
    let ofslpar = ofs + ofslpar in
    match i, x with
    | None, Some x ->
        overwrite content ofslpar '(' '_';
        overwrite content (ofslpar + 1 + String.length x) ')' '_'
    | Some i, None ->
        overwrite content ofslpar '(' '_';
        overwrite content (ofslpar + 1) '$' '_';
        overwrite content (ofslpar + 2 + String.length i) ')' '_'
    | _, _ ->
        ()
  in
  let keyword =
    Some (Position (subject, where, flavor))
  and oid =
    None
  in
  { pos; check; transform; keyword; oid }

(* ------------------------------------------------------------------------ *)

(* In an OCaml header, there should be no monsters. This is just a sanity
   check. *)

let no_monsters monsters =
  match monsters with
  | [] ->
      ()
  | monster :: _ ->
      Error.error [monster.pos]
        "a Menhir keyword cannot be used in an OCaml header."

(* ------------------------------------------------------------------------ *)

(* Gathering all of the identifiers in an array of optional identifiers. *)

let gather_oid xs oid =
  match oid with
  | Some x ->
      StringSet.add x xs
  | None ->
      xs

let gather_oids oids =
  Array.fold_left gather_oid StringSet.empty oids

(* Gathering all of the [oid] identifiers in a list of monsters. *)

let gather_monsters monsters =
  List.fold_left (fun xs monster ->
    gather_oid xs monster.oid
  ) StringSet.empty monsters

(* ------------------------------------------------------------------------ *)

(* Creates a stretch. *)

let mk_stretch pos1 pos2 parenthesize monsters =
  (* Read the specified chunk of the file. *)
  let raw_content : string = InputFile.chunk (pos1, pos2) in
  (* Transform the monsters, if there are any. (This explicit test
     allows saving one string copy and keeping just one live copy.) *)
  let content : string =
    match monsters with
    | [] ->
        raw_content
    | _ :: _ ->
        let content : bytes = Bytes.of_string raw_content in
        List.iter (fun monster -> monster.transform pos1.pos_cnum content) monsters;
        Bytes.unsafe_to_string content
  in
  (* Add whitespace so that the column numbers match those of the source file.
     If requested, add parentheses so that the semantic action can be inserted
     into other code without ambiguity. *)
  let content =
    if parenthesize then
      (* If [parenthesize] is true then we are at the beginning of a semantic
         action, just after the opening brace. This guarantees that we cannot
         be at the beginning of a line, so the subtraction [_ - 1] below
         cannot produce a negative result. *)
      (String.make (pos1.pos_cnum - pos1.pos_bol - 1) ' ') ^ "(" ^ content ^ ")"
    else
      (String.make (pos1.pos_cnum - pos1.pos_bol) ' ') ^ content
  in
  Stretch.({
    stretch_filename = InputFile.get_input_file_name();
    stretch_linenum = pos1.pos_lnum;
    stretch_linecount = pos2.pos_lnum - pos1.pos_lnum;
    stretch_content = content;
    stretch_raw_content = raw_content;
    stretch_keywords = Misc.filter_map (fun monster -> monster.keyword) monsters
  })

(* Creating a stretch from a located identifier. (This does not require the
   input file to be currently opened.) In this variant, [parenthesize] is
   false, [monsters] is empty. *)

let stretch_of_id (id : string located) =
  let raw_content, pos = Positions.decompose id in
  let pos1 = Positions.start_of_position pos
  and pos2 = Positions.end_of_position pos
  and filename = Positions.filename_of_position pos in
  assert (pos1 != Lexing.dummy_pos);
  let padding = pos1.pos_cnum - pos1.pos_bol in
  let content = String.make padding ' ' ^ raw_content in
  Stretch.({
    stretch_filename = filename;
    stretch_linenum = pos1.pos_lnum;
    stretch_linecount = pos2.pos_lnum - pos1.pos_lnum;
    stretch_content = content;
    stretch_raw_content = raw_content;
    stretch_keywords = []
  })

(* ------------------------------------------------------------------------ *)

(* OCaml's reserved words. *)

let table words =
  let table = Hashtbl.create 149 in
  List.iter (fun word -> Hashtbl.add table word ()) words;
  table

let reserved =
  table [
    "and";
    "as";
    "assert";
    "begin";
    "class";
    "constraint";
    "do";
    "done";
    "downto";
    "else";
    "end";
    "exception";
    "external";
    "false";
    "for";
    "fun";
    "function";
    "functor";
    "if";
    "in";
    "include";
    "inherit";
    "initializer";
    "lazy";
    "let";
    "match";
    "method";
    "module";
    "mutable";
    "new";
    "object";
    "of";
    "open";
    "or";
    "parser";
    "private";
    "rec";
    "sig";
    "struct";
    "then";
    "to";
    "true";
    "try";
    "type";
    "val";
    "virtual";
    "when";
    "while";
    "with";
    "mod";
    "land";
    "lor";
    "lxor";
    "lsl";
    "lsr";
    "asr";
  ]

(* ------------------------------------------------------------------------ *)

(* Menhir's percent-directives. *)

let table directives =
  let table = Hashtbl.create 149 in
  List.iter (fun (word, token) -> Hashtbl.add table word token) directives;
  table

let directives =
  table [
    "token", TOKEN;
    "type", TYPE;
    "left", LEFT;
    "right", RIGHT;
    "nonassoc", NONASSOC;
    "start", START;
    "prec", PREC;
    "public", PUBLIC;
    "parameter", PARAMETER;
    "inline", INLINE;
    "attribute", PERCENTATTRIBUTE;
    "on_error_reduce", ON_ERROR_REDUCE;
  ]


# 411 "src/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\226\255\227\255\000\000\229\255\231\255\003\000\001\000\
    \235\255\003\000\236\255\209\000\161\001\113\002\065\003\243\255\
    \244\255\245\255\246\255\247\255\248\255\004\000\001\000\251\255\
    \031\000\253\255\019\004\029\000\230\255\254\255\227\004\181\005\
    \163\006\006\000\241\255\240\255\232\255\115\007\067\008\233\255\
    \011\000\004\000\012\000\252\255\253\255\001\000\006\000\255\255\
    \254\255\124\000\250\255\251\255\007\000\252\255\007\000\254\255\
    \032\000\034\000\255\255\253\255\096\001\243\255\244\255\008\000\
    \245\255\247\255\248\255\236\001\009\000\000\000\254\255\255\255\
    \246\255\013\000\017\000\019\000\022\000\096\000\034\000\023\000\
    \019\000\038\000\028\000\032\000\026\000\026\000\057\000\041\000\
    \040\000\046\000\057\000\049\000\046\000\012\000\122\008\081\009\
    \122\000\035\001\251\255\251\255\047\000\066\000\045\000\065\000\
    \053\000\054\000\069\000\067\000\249\255\068\000\067\000\084\000\
    \086\000\086\000\070\000\084\000\079\000\074\000\132\000\168\000\
    \156\000\157\000\161\000\159\000\250\255\000\003\243\255\244\255\
    \009\000\245\255\247\255\248\255\140\003\253\255\254\255\011\000\
    \246\255\186\000\190\000\192\000\242\000\062\001\206\000\224\000\
    \189\000\237\000\225\000\229\000\223\000\233\000\254\000\238\000\
    \237\000\243\000\254\000\246\000\244\000\015\000\136\009\095\010\
    \246\001\007\002\251\255\251\255\245\000\008\001\244\000\019\001\
    \007\001\008\001\012\001\011\001\249\255\012\001\011\001\028\001\
    \030\001\030\001\016\001\033\001\028\001\023\001\026\001\041\001\
    \029\001\030\001\057\001\086\001\250\255\085\011\246\255\247\255\
    \010\000\248\255\250\255\251\255\014\000\253\255\254\255\255\255\
    \249\255\228\002\249\255\250\255\013\000\251\255\252\255\253\255\
    \015\000\179\001\255\255\254\255\188\003\251\255\252\255\016\000\
    \254\255\242\001\255\255\253\255\017\000\142\004\249\255\250\255\
    \018\000\251\255\065\011\255\255\252\255\253\255\254\255\236\002\
    \000\000\214\001\183\001\146\004\184\001\055\005\204\002\215\001\
    \218\001\255\255\010\002\017\002\254\255\011\002\000\006\012\002\
    \161\002\253\255\254\255\019\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\029\000\255\255\255\255\021\000\020\000\
    \255\255\029\000\255\255\018\000\017\000\017\000\013\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\005\000\255\255\
    \003\000\255\255\029\000\255\255\255\255\255\255\000\000\255\255\
    \027\000\027\000\255\255\255\255\255\255\017\000\016\000\255\255\
    \255\255\020\000\255\255\255\255\255\255\003\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\005\000\255\255\
    \005\000\005\000\255\255\255\255\255\255\255\255\255\255\010\000\
    \255\255\255\255\255\255\012\000\002\000\012\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\004\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \010\000\255\255\255\255\255\255\012\000\255\255\255\255\000\000\
    \255\255\255\255\255\255\255\255\255\255\003\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\004\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \007\000\255\255\255\255\255\255\003\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\004\000\255\255\255\255\255\255\
    \006\000\006\000\255\255\255\255\255\255\255\255\255\255\001\000\
    \255\255\004\000\255\255\255\255\001\000\255\255\255\255\255\255\
    \004\000\255\255\006\000\255\255\255\255\255\255\255\255\001\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\000\000\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\000\000\000\000\255\255\255\255\
    \000\000\255\255\000\000\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\255\255\255\255\000\000\
    \255\255\000\000\255\255\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\000\000\000\000\000\000\255\255\255\255\000\000\
    \040\000\255\255\043\000\000\000\000\000\255\255\255\255\000\000\
    \000\000\050\000\000\000\000\000\255\255\000\000\255\255\000\000\
    \255\255\255\255\000\000\000\000\061\000\000\000\000\000\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\126\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\190\000\000\000\000\000\
    \255\255\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\202\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \255\255\255\255\000\000\000\000\213\000\000\000\000\000\255\255\
    \000\000\219\000\000\000\000\000\255\255\222\000\000\000\000\000\
    \255\255\000\000\228\000\000\000\000\000\000\000\000\000\232\000\
    \255\255\255\255\255\255\236\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \249\000\000\000\000\000\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\008\000\008\000\006\000\007\000\008\000\033\000\
    \047\000\053\000\064\000\129\000\193\000\008\000\047\000\205\000\
    \041\000\046\000\216\000\216\000\225\000\252\000\000\000\000\000\
    \006\000\000\000\010\000\006\000\000\000\026\000\033\000\244\000\
    \021\000\020\000\017\000\016\000\023\000\039\000\036\000\009\000\
    \048\000\059\000\040\000\072\000\094\000\136\000\045\000\158\000\
    \200\000\211\000\024\000\025\000\005\000\022\000\035\000\018\000\
    \031\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\003\000\034\000\031\000\058\000\014\000\
    \058\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\013\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\004\000\019\000\070\000\015\000\113\000\
    \112\000\111\000\074\000\109\000\081\000\080\000\053\000\101\000\
    \082\000\052\000\078\000\083\000\084\000\085\000\086\000\079\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\087\000\088\000\089\000\090\000\091\000\100\000\
    \092\000\093\000\093\000\102\000\054\000\103\000\104\000\105\000\
    \106\000\057\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\107\000\108\000\110\000\089\000\
    \089\000\093\000\055\000\114\000\115\000\116\000\117\000\118\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\056\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\119\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \002\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\255\255\044\000\120\000\121\000\122\000\
    \123\000\124\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\177\000\176\000\175\000\173\000\
    \011\000\165\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\098\000\145\000\144\000\146\000\
    \147\000\148\000\149\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\150\000\138\000\151\000\
    \152\000\153\000\154\000\155\000\164\000\156\000\142\000\157\000\
    \157\000\166\000\064\000\143\000\167\000\063\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \168\000\169\000\170\000\171\000\051\000\172\000\174\000\153\000\
    \153\000\157\000\066\000\178\000\067\000\069\000\179\000\065\000\
    \068\000\062\000\180\000\181\000\182\000\183\000\184\000\185\000\
    \186\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \187\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \188\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\071\000\210\000\070\000\241\000\244\000\
    \242\000\242\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\216\000\241\000\241\000\220\000\
    \012\000\241\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \162\000\241\000\244\000\244\000\000\000\000\000\000\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\245\000\245\000\245\000\245\000\245\000\245\000\245\000\
    \245\000\245\000\245\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\075\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \074\000\000\000\000\000\000\000\073\000\000\000\000\000\076\000\
    \062\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\252\000\000\000\000\000\251\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\000\000\012\000\012\000\012\000\012\000\037\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\205\000\000\000\
    \000\000\204\000\255\255\244\000\000\000\000\000\234\000\000\000\
    \000\000\233\000\000\000\000\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\243\000\243\000\243\000\207\000\000\000\
    \000\000\000\000\129\000\206\000\208\000\128\000\209\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\131\000\000\000\132\000\000\000\000\000\130\000\
    \135\000\134\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \235\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \000\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\133\000\000\000\127\000\000\000\000\000\
    \000\000\000\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\000\000\000\000\000\000\000\000\
    \012\000\250\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\141\000\141\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\216\000\000\000\
    \000\000\215\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\218\000\000\000\
    \000\000\000\000\000\000\000\000\203\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\139\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \138\000\000\000\000\000\000\000\137\000\000\000\000\000\140\000\
    \127\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \217\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \029\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\027\000\000\000\
    \000\000\000\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\028\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \225\000\000\000\000\000\224\000\240\000\000\000\000\000\239\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \227\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\214\000\000\000\000\000\000\000\
    \000\000\000\000\238\000\238\000\238\000\238\000\238\000\238\000\
    \238\000\238\000\238\000\238\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\226\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\237\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\000\000\000\000\
    \000\000\000\000\030\000\000\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\244\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\246\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\246\000\246\000\
    \246\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\223\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\000\000\000\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\000\000\000\000\
    \000\000\000\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \000\000\000\000\000\000\000\000\032\000\000\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \247\000\247\000\247\000\247\000\247\000\247\000\247\000\247\000\
    \247\000\247\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\247\000\247\000\247\000\247\000\247\000\247\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\247\000\247\000\247\000\247\000\247\000\247\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\000\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\033\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\033\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\000\000\032\000\000\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\012\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\038\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\096\000\000\000\
    \000\000\000\000\012\000\000\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\000\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\000\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\099\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\160\000\000\000\000\000\000\000\
    \095\000\000\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \000\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \000\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \163\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\000\000\000\000\000\000\000\000\159\000\000\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\000\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\193\000\
    \000\000\230\000\192\000\230\000\000\000\000\000\000\000\000\000\
    \230\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\195\000\
    \000\000\000\000\000\000\000\000\194\000\196\000\191\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\230\000\000\000\000\000\
    \000\000\000\000\000\000\230\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\229\000\
    \199\000\000\000\198\000\230\000\000\000\230\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \197\000\000\000\191\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\191\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\007\000\006\000\000\000\041\000\033\000\
    \046\000\052\000\063\000\128\000\192\000\040\000\042\000\204\000\
    \040\000\042\000\215\000\220\000\224\000\251\000\255\255\255\255\
    \000\000\255\255\000\000\006\000\255\255\000\000\033\000\232\000\
    \000\000\000\000\000\000\000\000\000\000\009\000\021\000\000\000\
    \045\000\054\000\009\000\068\000\093\000\135\000\042\000\157\000\
    \196\000\208\000\000\000\000\000\000\000\000\000\022\000\000\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\024\000\027\000\056\000\000\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\069\000\000\000\073\000\
    \074\000\075\000\076\000\078\000\079\000\079\000\049\000\080\000\
    \081\000\049\000\076\000\082\000\083\000\084\000\085\000\076\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\086\000\087\000\088\000\089\000\089\000\090\000\
    \091\000\092\000\100\000\101\000\049\000\102\000\103\000\104\000\
    \105\000\049\000\096\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\106\000\107\000\109\000\110\000\
    \111\000\112\000\049\000\113\000\114\000\115\000\116\000\117\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\049\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\118\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\040\000\042\000\119\000\120\000\121\000\
    \122\000\123\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\137\000\138\000\139\000\142\000\
    \011\000\144\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\097\000\143\000\143\000\145\000\
    \146\000\147\000\148\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\097\000\097\000\097\000\149\000\140\000\150\000\
    \151\000\152\000\153\000\153\000\154\000\155\000\140\000\156\000\
    \164\000\165\000\060\000\140\000\166\000\060\000\141\000\141\000\
    \141\000\141\000\141\000\141\000\141\000\141\000\141\000\141\000\
    \167\000\168\000\169\000\170\000\049\000\171\000\173\000\174\000\
    \175\000\176\000\060\000\177\000\060\000\060\000\178\000\060\000\
    \060\000\060\000\179\000\180\000\181\000\182\000\183\000\184\000\
    \185\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \186\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \187\000\011\000\011\000\011\000\011\000\011\000\011\000\011\000\
    \011\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\060\000\209\000\060\000\234\000\236\000\
    \233\000\239\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\217\000\233\000\239\000\217\000\
    \012\000\240\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \161\000\242\000\245\000\247\000\255\255\255\255\255\255\161\000\
    \161\000\161\000\161\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\243\000\243\000\243\000\243\000\243\000\243\000\243\000\
    \243\000\243\000\243\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\067\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \067\000\255\255\255\255\255\255\067\000\255\255\255\255\067\000\
    \060\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \255\255\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \255\255\012\000\012\000\012\000\012\000\012\000\012\000\012\000\
    \012\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\248\000\255\255\255\255\248\000\255\255\
    \255\255\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\255\255\255\255\255\255\255\255\
    \013\000\255\255\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\255\255\255\255\201\000\255\255\
    \255\255\201\000\217\000\238\000\255\255\255\255\231\000\255\255\
    \255\255\231\000\255\255\255\255\238\000\238\000\238\000\238\000\
    \238\000\238\000\238\000\238\000\238\000\238\000\201\000\255\255\
    \255\255\255\255\125\000\201\000\201\000\125\000\201\000\255\255\
    \255\255\255\255\255\255\231\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\125\000\255\255\125\000\255\255\255\255\125\000\
    \125\000\125\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \231\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \255\255\013\000\013\000\013\000\013\000\013\000\013\000\013\000\
    \013\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\125\000\255\255\125\000\255\255\255\255\
    \255\255\255\255\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\255\255\255\255\255\255\255\255\
    \014\000\248\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\132\000\132\000\132\000\132\000\
    \132\000\132\000\132\000\132\000\132\000\132\000\212\000\255\255\
    \255\255\212\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\212\000\255\255\
    \255\255\255\255\255\255\255\255\201\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\231\000\255\255\255\255\255\255\
    \255\255\132\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \132\000\255\255\255\255\255\255\132\000\255\255\255\255\132\000\
    \125\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \212\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \026\000\014\000\014\000\014\000\014\000\014\000\014\000\014\000\
    \014\000\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \255\255\255\255\026\000\255\255\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \221\000\255\255\255\255\221\000\235\000\255\255\255\255\235\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \221\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\212\000\255\255\255\255\255\255\
    \255\255\255\255\235\000\235\000\235\000\235\000\235\000\235\000\
    \235\000\235\000\235\000\235\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\221\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\235\000\026\000\026\000\026\000\026\000\026\000\
    \026\000\026\000\026\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\255\255\255\255\
    \255\255\255\255\030\000\255\255\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\237\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\237\000\
    \237\000\237\000\237\000\237\000\237\000\237\000\237\000\237\000\
    \237\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \237\000\237\000\237\000\237\000\237\000\237\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\221\000\255\255\
    \255\255\255\255\235\000\255\255\255\255\255\255\255\255\255\255\
    \237\000\237\000\237\000\237\000\237\000\237\000\255\255\255\255\
    \255\255\255\255\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \255\255\255\255\255\255\255\255\031\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \246\000\246\000\246\000\246\000\246\000\246\000\246\000\246\000\
    \246\000\246\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\246\000\246\000\246\000\246\000\246\000\246\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\246\000\246\000\246\000\246\000\246\000\246\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\255\255\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\032\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\032\000\255\255\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\255\255\
    \255\255\255\255\037\000\255\255\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\094\000\255\255\
    \255\255\255\255\038\000\255\255\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\094\000\255\255\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\255\255\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\158\000\255\255\255\255\255\255\
    \095\000\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\255\255\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \159\000\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\189\000\
    \255\255\226\000\189\000\226\000\255\255\255\255\255\255\255\255\
    \226\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\189\000\
    \255\255\255\255\255\255\255\255\189\000\189\000\189\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\226\000\255\255\255\255\
    \255\255\255\255\255\255\226\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\226\000\
    \189\000\255\255\189\000\226\000\255\255\226\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \189\000\255\255\189\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\226\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\189\000";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \210\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\008\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \008\000\000\000\000\000\000\000\010\000\000\000\009\001\217\001\
    \078\000\088\000\052\000\067\000\031\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\031\000\048\000\
    \050\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\002\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\035\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\000\000\000\000\000\000\037\000\001\000\016\002\224\002\
    \105\000\115\000\052\000\067\000\058\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\058\000\075\000\
    \077\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \007\000\007\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\027\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\027\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\012\000\001\000\012\000\000\000\
    \040\000\040\000\000\000\000\000\000\000\000\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\000\000\000\000\000\000\000\000\004\000\
    \000\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\019\000\019\000\024\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\024\000\019\000\019\000\024\000\019\000\019\000\
    \024\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\024\000\019\000\019\000\
    \024\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\043\000\000\000\000\000\
    \000\000\004\000\000\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\000\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\043\000\000\000\000\000\000\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\046\000\
    \000\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \046\000\046\000\046\000\046\000\046\000\046\000\046\000\046\000\
    \000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\060\000\000\000\125\000\255\255\
    \093\000\157\000\255\255\255\255\255\255\255\255\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\255\255\255\255\255\255\255\255\031\000\
    \255\255\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\067\000\088\000\092\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\096\000\096\000\096\000\096\000\
    \097\000\097\000\097\000\097\000\097\000\097\000\097\000\097\000\
    \097\000\097\000\100\000\110\000\111\000\112\000\132\000\152\000\
    \156\000\160\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\161\000\161\000\161\000\161\000\161\000\
    \161\000\161\000\161\000\161\000\161\000\164\000\174\000\175\000\
    \176\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\255\255\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\094\000\255\255\255\255\
    \255\255\032\000\255\255\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \094\000\255\255\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \255\255\094\000\094\000\094\000\094\000\094\000\094\000\094\000\
    \094\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\158\000\255\255\255\255\255\255\
    \095\000\255\255\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \255\255\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \255\255\095\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\255\255\
    \158\000\158\000\158\000\158\000\158\000\158\000\158\000\158\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\255\255\255\255\255\255\159\000\
    \255\255\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\255\255\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \255\255";
  Lexing.lex_code =
   "\255\002\255\255\003\255\255\000\002\001\003\255\010\255\009\255\
    \008\255\255\010\255\009\255\255\011\255\255\007\255\003\255\005\
    \008\004\010\001\009\000\011\255\012\255\255\014\255\255\013\255\
    \255\015\255\255\007\255\005\008\004\010\003\014\002\015\001\009\
    \000\011\255\003\255\007\012\006\013\005\008\004\010\001\009\000\
    \011\255";
}

let rec main lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 4 (-1);(* L=1 [2] <- p ;  *)
  lexbuf.Lexing.lex_mem.(2) <- lexbuf.Lexing.lex_curr_pos ;
 __ocaml_lex_main_rec lexbuf 0
and __ocaml_lex_main_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 459 "src/lexer.mll"
                     directive
# 1674 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 460 "src/lexer.mll"
    ( try Hashtbl.find directives directive
      with Not_found -> error2 lexbuf "unknown directive: %s." directive )
# 1679 "src/lexer.ml"

  | 1 ->
# 463 "src/lexer.mll"
    ( (* The token [PERCENTPERCENT] carries a stretch that contains
         everything that follows %% in the input file. This string
         must be created lazily. The parser decides (based on the
         context) whether this stretch is needed. If it is indeed
         needed, then constructing this stretch drives the lexer
         to the end of the file. *)
      PERCENTPERCENT (lazy (
        let openingpos = lexeme_end_p lexbuf in
        let closingpos = finish lexbuf in
        mk_stretch openingpos closingpos false []
      )) )
# 1694 "src/lexer.ml"

  | 2 ->
# 475 "src/lexer.mll"
    ( SEMI )
# 1699 "src/lexer.ml"

  | 3 ->
# 477 "src/lexer.mll"
    ( COLON )
# 1704 "src/lexer.ml"

  | 4 ->
# 479 "src/lexer.mll"
    ( COMMA )
# 1709 "src/lexer.ml"

  | 5 ->
# 481 "src/lexer.mll"
    ( EQUAL )
# 1714 "src/lexer.ml"

  | 6 ->
# 483 "src/lexer.mll"
    ( LPAREN )
# 1719 "src/lexer.ml"

  | 7 ->
# 485 "src/lexer.mll"
    ( RPAREN )
# 1724 "src/lexer.ml"

  | 8 ->
# 487 "src/lexer.mll"
    ( BAR )
# 1729 "src/lexer.ml"

  | 9 ->
# 489 "src/lexer.mll"
    ( QUESTION )
# 1734 "src/lexer.ml"

  | 10 ->
# 491 "src/lexer.mll"
    ( STAR )
# 1739 "src/lexer.ml"

  | 11 ->
# 493 "src/lexer.mll"
    ( PLUS )
# 1744 "src/lexer.ml"

  | 12 ->
# 495 "src/lexer.mll"
    ( TILDE )
# 1749 "src/lexer.ml"

  | 13 ->
# 497 "src/lexer.mll"
    ( UNDERSCORE )
# 1754 "src/lexer.ml"

  | 14 ->
# 499 "src/lexer.mll"
    ( COLONEQUAL )
# 1759 "src/lexer.ml"

  | 15 ->
# 501 "src/lexer.mll"
    ( EQUALEQUAL )
# 1764 "src/lexer.ml"

  | 16 ->
# 503 "src/lexer.mll"
    ( LET )
# 1769 "src/lexer.ml"

  | 17 ->
let
# 504 "src/lexer.mll"
                             id
# 1775 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 505 "src/lexer.mll"
    ( if Hashtbl.mem reserved id then
        error2 lexbuf "this is an OCaml reserved word."
      else
        LID (with_pos (cpos lexbuf) id)
    )
# 1783 "src/lexer.ml"

  | 18 ->
let
# 510 "src/lexer.mll"
                             id
# 1789 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 511 "src/lexer.mll"
    ( UID (with_pos (cpos lexbuf) id) )
# 1793 "src/lexer.ml"

  | 19 ->
# 516 "src/lexer.mll"
    ( let buffer = Buffer.create 16 in
      let openingpos = lexeme_start_p lexbuf in
      let content = record_string openingpos buffer lexbuf in
      let id = Printf.sprintf "\"%s\"" content in
      let pos = import (openingpos, lexbuf.lex_curr_p) in
      QID (with_pos pos id) )
# 1803 "src/lexer.ml"

  | 20 ->
# 524 "src/lexer.mll"
    ( new_line lexbuf; main lexbuf )
# 1808 "src/lexer.ml"

  | 21 ->
# 526 "src/lexer.mll"
    ( main lexbuf )
# 1813 "src/lexer.ml"

  | 22 ->
# 528 "src/lexer.mll"
    ( comment (lexeme_start_p lexbuf) lexbuf; main lexbuf )
# 1818 "src/lexer.ml"

  | 23 ->
# 530 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf; main lexbuf )
# 1823 "src/lexer.ml"

  | 24 ->
# 532 "src/lexer.mll"
    ( savestart lexbuf (ocamltype (lexeme_end_p lexbuf)) )
# 1828 "src/lexer.ml"

  | 25 ->
# 534 "src/lexer.mll"
    ( savestart lexbuf (fun lexbuf ->
        let openingpos = lexeme_start_p lexbuf in
        let stretchpos = lexeme_end_p lexbuf in
        let closingpos, monsters = action true openingpos [] lexbuf in
        no_monsters monsters;
        HEADER (mk_stretch stretchpos closingpos false [])
      ) )
# 1839 "src/lexer.ml"

  | 26 ->
# 542 "src/lexer.mll"
    ( savestart lexbuf (fun lexbuf ->
        let openingpos = lexeme_start_p lexbuf in
        let stretchpos = lexeme_end_p lexbuf in
        let closingpos, monsters = action false openingpos [] lexbuf in
        ACTION (
          fun dollars producers ->
            (* Check that the monsters are well-formed. *)
            List.iter (fun monster -> monster.check dollars producers) monsters;
            (* Gather all of the identifiers that the semantic action may use
               to refer to a semantic value. This includes the identifiers
               that are explicitly bound by the user (these appear in the
               array [producers]) and the identifiers [_i] when the semantic
               action uses [$i]. *)
            let ids =
              StringSet.union (gather_oids producers) (gather_monsters monsters)
            in
            (* Extract a stretch of text. *)
            let stretch = mk_stretch stretchpos closingpos true monsters in
            (* Build a semantic action. *)
            Action.from_stretch ids stretch
        )
      )
    )
# 1866 "src/lexer.ml"

  | 27 ->
let
# 565 "src/lexer.mll"
           percent
# 1872 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 565 "src/lexer.mll"
                                            key
# 1877 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_mem.(0) + 2) lexbuf.Lexing.lex_mem.(1) in
# 566 "src/lexer.mll"
    ( let openingpos = lexeme_start_p lexbuf in
      let stretchpos = lexeme_end_p lexbuf in
      let closingpos = attribute openingpos lexbuf in
      let origin = Positions.import (openingpos, lexeme_end_p lexbuf) in
      let payload = InputFile.chunk (stretchpos, closingpos) in
      let attr : attribute = { key; payload; origin } in
      if percent = "" then
        (* No [%] sign: this is a normal attribute. *)
        ATTRIBUTE attr
      else
        (* A [%] sign is present: this is a grammar-wide attribute. *)
        GRAMMARATTRIBUTE attr
    )
# 1893 "src/lexer.ml"

  | 28 ->
# 580 "src/lexer.mll"
    ( EOF )
# 1898 "src/lexer.ml"

  | 29 ->
# 582 "src/lexer.mll"
    ( error2 lexbuf "unexpected character(s)." )
# 1903 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_main_rec lexbuf __ocaml_lex_state

and comment openingpos lexbuf =
   __ocaml_lex_comment_rec openingpos lexbuf 42
and __ocaml_lex_comment_rec openingpos lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 590 "src/lexer.mll"
    ( new_line lexbuf; comment openingpos lexbuf )
# 1915 "src/lexer.ml"

  | 1 ->
# 592 "src/lexer.mll"
    ( () )
# 1920 "src/lexer.ml"

  | 2 ->
# 594 "src/lexer.mll"
    ( error1 openingpos "unterminated comment." )
# 1925 "src/lexer.ml"

  | 3 ->
# 596 "src/lexer.mll"
    ( comment openingpos lexbuf )
# 1930 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec openingpos lexbuf __ocaml_lex_state

and ocamltype openingpos lexbuf =
   __ocaml_lex_ocamltype_rec openingpos lexbuf 49
and __ocaml_lex_ocamltype_rec openingpos lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 607 "src/lexer.mll"
    ( ocamltype openingpos lexbuf )
# 1942 "src/lexer.ml"

  | 1 ->
# 609 "src/lexer.mll"
    ( OCAMLTYPE (Stretch.Declared (mk_stretch openingpos (lexeme_start_p lexbuf) true [])) )
# 1947 "src/lexer.ml"

  | 2 ->
# 611 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf; ocamltype openingpos lexbuf )
# 1952 "src/lexer.ml"

  | 3 ->
# 613 "src/lexer.mll"
    ( new_line lexbuf; ocamltype openingpos lexbuf )
# 1957 "src/lexer.ml"

  | 4 ->
# 615 "src/lexer.mll"
    ( error1 openingpos "unterminated OCaml type." )
# 1962 "src/lexer.ml"

  | 5 ->
# 617 "src/lexer.mll"
    ( ocamltype openingpos lexbuf )
# 1967 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_ocamltype_rec openingpos lexbuf __ocaml_lex_state

and action percent openingpos monsters lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 16 (-1); __ocaml_lex_action_rec percent openingpos monsters lexbuf 60
and __ocaml_lex_action_rec percent openingpos monsters lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 628 "src/lexer.mll"
    ( let _, monsters = action false (lexeme_start_p lexbuf) monsters lexbuf in
      action percent openingpos monsters lexbuf )
# 1980 "src/lexer.ml"

  | 1 ->
let
# 630 "src/lexer.mll"
                  delimiter
# 1986 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 631 "src/lexer.mll"
    ( match percent, delimiter with
      | true, "%}"
      | false, "}" ->
          (* This is the delimiter we were instructed to look for. *)
          lexeme_start_p lexbuf, monsters
      | _, _ ->
          (* This is not it. *)
          error1 openingpos "unbalanced opening brace."
    )
# 1998 "src/lexer.ml"

  | 2 ->
# 641 "src/lexer.mll"
    ( let _, monsters = parentheses (lexeme_start_p lexbuf) monsters lexbuf in
      action percent openingpos monsters lexbuf )
# 2004 "src/lexer.ml"

  | 3 ->
let
# 643 "src/lexer.mll"
                     i
# 2010 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 644 "src/lexer.mll"
    ( let i = int_of_string (lexeme_start_p lexbuf) i in
      let monster = dollar (cpos lexbuf) i in
      action percent openingpos (monster :: monsters) lexbuf )
# 2016 "src/lexer.ml"

  | 4 ->
let
# 437 "src/lexer.mll"
                     i
# 2022 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2)
and
# 438 "src/lexer.mll"
                             x
# 2027 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(7) lexbuf.Lexing.lex_mem.(6)
and
# 443 "src/lexer.mll"
                                          where
# 2032 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 443 "src/lexer.mll"
                                                                     flavor
# 2037 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 648 "src/lexer.mll"
    ( let monster = position (cpos lexbuf) where flavor i x in
      action percent openingpos (monster :: monsters) lexbuf )
# 2042 "src/lexer.ml"

  | 5 ->
# 651 "src/lexer.mll"
    ( error2 lexbuf "$previouserror is no longer supported." )
# 2047 "src/lexer.ml"

  | 6 ->
# 653 "src/lexer.mll"
    ( error2 lexbuf "$syntaxerror is no longer supported." )
# 2052 "src/lexer.ml"

  | 7 ->
# 655 "src/lexer.mll"
    ( string (lexeme_start_p lexbuf) lexbuf;
      action percent openingpos monsters lexbuf )
# 2058 "src/lexer.ml"

  | 8 ->
# 658 "src/lexer.mll"
    ( char lexbuf;
      action percent openingpos monsters lexbuf )
# 2064 "src/lexer.ml"

  | 9 ->
# 661 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf;
      action percent openingpos monsters lexbuf )
# 2070 "src/lexer.ml"

  | 10 ->
# 664 "src/lexer.mll"
    ( new_line lexbuf;
      action percent openingpos monsters lexbuf )
# 2076 "src/lexer.ml"

  | 11 ->
# 668 "src/lexer.mll"
    ( error1 openingpos "unbalanced opening brace." )
# 2081 "src/lexer.ml"

  | 12 ->
# 670 "src/lexer.mll"
    ( action percent openingpos monsters lexbuf )
# 2086 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_action_rec percent openingpos monsters lexbuf __ocaml_lex_state

and parentheses openingpos monsters lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 16 (-1); __ocaml_lex_parentheses_rec openingpos monsters lexbuf 125
and __ocaml_lex_parentheses_rec openingpos monsters lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 679 "src/lexer.mll"
    ( let _, monsters = parentheses (lexeme_start_p lexbuf) monsters lexbuf in
      parentheses openingpos monsters lexbuf )
# 2099 "src/lexer.ml"

  | 1 ->
# 682 "src/lexer.mll"
    ( lexeme_start_p lexbuf, monsters )
# 2104 "src/lexer.ml"

  | 2 ->
# 684 "src/lexer.mll"
    ( let _, monsters = action false (lexeme_start_p lexbuf) monsters lexbuf in
      parentheses openingpos monsters lexbuf )
# 2110 "src/lexer.ml"

  | 3 ->
let
# 686 "src/lexer.mll"
                     i
# 2116 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 687 "src/lexer.mll"
    ( let i = int_of_string (lexeme_start_p lexbuf) i in
      let monster = dollar (cpos lexbuf) i in
      parentheses openingpos (monster :: monsters) lexbuf )
# 2122 "src/lexer.ml"

  | 4 ->
let
# 437 "src/lexer.mll"
                     i
# 2128 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(3) lexbuf.Lexing.lex_mem.(2)
and
# 438 "src/lexer.mll"
                             x
# 2133 "src/lexer.ml"
= Lexing.sub_lexeme_opt lexbuf lexbuf.Lexing.lex_mem.(7) lexbuf.Lexing.lex_mem.(6)
and
# 443 "src/lexer.mll"
                                          where
# 2138 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(5) lexbuf.Lexing.lex_mem.(4)
and
# 443 "src/lexer.mll"
                                                                     flavor
# 2143 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(1) lexbuf.Lexing.lex_mem.(0) in
# 691 "src/lexer.mll"
    ( let monster = position (cpos lexbuf) where flavor i x in
      parentheses openingpos (monster :: monsters) lexbuf )
# 2148 "src/lexer.ml"

  | 5 ->
# 694 "src/lexer.mll"
    ( error2 lexbuf "$previouserror is no longer supported." )
# 2153 "src/lexer.ml"

  | 6 ->
# 696 "src/lexer.mll"
    ( error2 lexbuf "$syntaxerror is no longer supported." )
# 2158 "src/lexer.ml"

  | 7 ->
# 698 "src/lexer.mll"
    ( string (lexeme_start_p lexbuf) lexbuf; parentheses openingpos monsters lexbuf )
# 2163 "src/lexer.ml"

  | 8 ->
# 700 "src/lexer.mll"
    ( char lexbuf; parentheses openingpos monsters lexbuf )
# 2168 "src/lexer.ml"

  | 9 ->
# 702 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf; parentheses openingpos monsters lexbuf )
# 2173 "src/lexer.ml"

  | 10 ->
# 704 "src/lexer.mll"
    ( new_line lexbuf; parentheses openingpos monsters lexbuf )
# 2178 "src/lexer.ml"

  | 11 ->
# 707 "src/lexer.mll"
    ( error1 openingpos "unbalanced opening parenthesis." )
# 2183 "src/lexer.ml"

  | 12 ->
# 709 "src/lexer.mll"
    ( parentheses openingpos monsters lexbuf )
# 2188 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_parentheses_rec openingpos monsters lexbuf __ocaml_lex_state

and attribute openingpos lexbuf =
   __ocaml_lex_attribute_rec openingpos lexbuf 189
and __ocaml_lex_attribute_rec openingpos lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 720 "src/lexer.mll"
    ( let _ = attribute (lexeme_start_p lexbuf) lexbuf in
      attribute openingpos lexbuf )
# 2201 "src/lexer.ml"

  | 1 ->
# 723 "src/lexer.mll"
    ( lexeme_start_p lexbuf )
# 2206 "src/lexer.ml"

  | 2 ->
# 725 "src/lexer.mll"
    ( let _, _ = action false (lexeme_start_p lexbuf) [] lexbuf in
      attribute openingpos lexbuf )
# 2212 "src/lexer.ml"

  | 3 ->
# 728 "src/lexer.mll"
    ( let _, _ = parentheses (lexeme_start_p lexbuf) [] lexbuf in
      attribute openingpos lexbuf )
# 2218 "src/lexer.ml"

  | 4 ->
# 731 "src/lexer.mll"
    ( string (lexeme_start_p lexbuf) lexbuf; attribute openingpos lexbuf )
# 2223 "src/lexer.ml"

  | 5 ->
# 733 "src/lexer.mll"
    ( char lexbuf; attribute openingpos lexbuf )
# 2228 "src/lexer.ml"

  | 6 ->
# 735 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf; attribute openingpos lexbuf )
# 2233 "src/lexer.ml"

  | 7 ->
# 737 "src/lexer.mll"
    ( new_line lexbuf; attribute openingpos lexbuf )
# 2238 "src/lexer.ml"

  | 8 ->
# 741 "src/lexer.mll"
    ( error1 openingpos "unbalanced opening bracket." )
# 2243 "src/lexer.ml"

  | 9 ->
# 743 "src/lexer.mll"
    ( attribute openingpos lexbuf )
# 2248 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_attribute_rec openingpos lexbuf __ocaml_lex_state

and ocamlcomment openingpos lexbuf =
   __ocaml_lex_ocamlcomment_rec openingpos lexbuf 201
and __ocaml_lex_ocamlcomment_rec openingpos lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 755 "src/lexer.mll"
    ( () )
# 2260 "src/lexer.ml"

  | 1 ->
# 757 "src/lexer.mll"
    ( ocamlcomment (lexeme_start_p lexbuf) lexbuf; ocamlcomment openingpos lexbuf )
# 2265 "src/lexer.ml"

  | 2 ->
# 759 "src/lexer.mll"
    ( string (lexeme_start_p lexbuf) lexbuf; ocamlcomment openingpos lexbuf )
# 2270 "src/lexer.ml"

  | 3 ->
# 761 "src/lexer.mll"
    ( char lexbuf; ocamlcomment openingpos lexbuf )
# 2275 "src/lexer.ml"

  | 4 ->
# 763 "src/lexer.mll"
    ( new_line lexbuf; ocamlcomment openingpos lexbuf )
# 2280 "src/lexer.ml"

  | 5 ->
# 765 "src/lexer.mll"
    ( error1 openingpos "unterminated OCaml comment." )
# 2285 "src/lexer.ml"

  | 6 ->
# 767 "src/lexer.mll"
    ( ocamlcomment openingpos lexbuf )
# 2290 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_ocamlcomment_rec openingpos lexbuf __ocaml_lex_state

and string openingpos lexbuf =
   __ocaml_lex_string_rec openingpos lexbuf 212
and __ocaml_lex_string_rec openingpos lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 775 "src/lexer.mll"
    ( () )
# 2302 "src/lexer.ml"

  | 1 ->
# 778 "src/lexer.mll"
    ( new_line lexbuf; string openingpos lexbuf )
# 2307 "src/lexer.ml"

  | 2 ->
# 782 "src/lexer.mll"
    ( string openingpos lexbuf )
# 2312 "src/lexer.ml"

  | 3 ->
# 784 "src/lexer.mll"
    ( error1 openingpos "unterminated OCaml string." )
# 2317 "src/lexer.ml"

  | 4 ->
# 786 "src/lexer.mll"
    ( string openingpos lexbuf )
# 2322 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec openingpos lexbuf __ocaml_lex_state

and record_string openingpos buffer lexbuf =
   __ocaml_lex_record_string_rec openingpos buffer lexbuf 221
and __ocaml_lex_record_string_rec openingpos buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 794 "src/lexer.mll"
    ( Buffer.contents buffer )
# 2334 "src/lexer.ml"

  | 1 ->
let
# 795 "src/lexer.mll"
                                            sequence
# 2340 "src/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_start_pos + 2) in
# 796 "src/lexer.mll"
    ( (* This escape sequence is recognized as such, but not decoded. *)
      Buffer.add_string buffer sequence;
      record_string openingpos buffer lexbuf )
# 2346 "src/lexer.ml"

  | 2 ->
# 804 "src/lexer.mll"
    ( error2 lexbuf "'\\n' is not permitted in a token alias." )
# 2351 "src/lexer.ml"

  | 3 ->
# 806 "src/lexer.mll"
    ( error2 lexbuf "illegal backslash escape in string." )
# 2356 "src/lexer.ml"

  | 4 ->
# 808 "src/lexer.mll"
    ( error2 lexbuf "illegal newline in string." )
# 2361 "src/lexer.ml"

  | 5 ->
# 810 "src/lexer.mll"
    ( error1 openingpos "unterminated string." )
# 2366 "src/lexer.ml"

  | 6 ->
let
# 811 "src/lexer.mll"
       c
# 2372 "src/lexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 812 "src/lexer.mll"
    ( Buffer.add_char buffer c;
      record_string openingpos buffer lexbuf )
# 2377 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_record_string_rec openingpos buffer lexbuf __ocaml_lex_state

and char lexbuf =
   __ocaml_lex_char_rec lexbuf 231
and __ocaml_lex_char_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 823 "src/lexer.mll"
   ( new_line lexbuf )
# 2389 "src/lexer.ml"

  | 1 ->
# 829 "src/lexer.mll"
   ( () )
# 2394 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_char_rec lexbuf __ocaml_lex_state

and finish lexbuf =
   __ocaml_lex_finish_rec lexbuf 248
and __ocaml_lex_finish_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 840 "src/lexer.mll"
    ( new_line lexbuf; finish lexbuf )
# 2406 "src/lexer.ml"

  | 1 ->
# 842 "src/lexer.mll"
    ( lexeme_start_p lexbuf )
# 2411 "src/lexer.ml"

  | 2 ->
# 844 "src/lexer.mll"
    ( finish lexbuf )
# 2416 "src/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_finish_rec lexbuf __ocaml_lex_state

;;

