import { objectToSnake, objectToCamel } from 'ts-case-convert';
import { getDefaultClient } from '../client.js';

function buildRequest(path, method) {
  const _request = async (client, init, mapResult) => {
    const { data, error } = await client[method.toUpperCase()](path, init);
    if (error || !data) {
      throw new Error(
        [
          "Request failed:",
          `path: ${path}`,
          `error: ${error}`,
          `data: ${data}`
        ].join("\n  ")
      );
    }
    const result = objectToCamel(data);
    return mapResult?.(result) ?? result;
  };
  return {
    withParams(getInit, mapResult) {
      return async (params, init, client = getDefaultClient()) => _request(client, { ...getInit(params), ...init }, mapResult);
    },
    withoutParams(mapResult) {
      return async (init, client = getDefaultClient()) => _request(client, init, mapResult);
    }
  };
}
const getActivity = buildRequest(
  "/decentralized/tx/{id}",
  "get"
).withParams(({ id, ...query }) => ({
  params: objectToSnake({ path: { id }, query })
}));
const getActivities = buildRequest(
  "/decentralized/{account}",
  "get"
).withParams(
  ({ account, ...query }) => ({
    params: objectToSnake({ path: { account }, query })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getActivitiesByAccounts = buildRequest(
  "/decentralized/accounts",
  "post"
).withParams(
  (body) => ({
    body: objectToSnake({
      limit: 20,
      actionLimit: 10,
      ...body
    })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getRSSActivity = buildRequest("/rss/{path}", "get").withParams(
  (path) => ({
    params: objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getNetworkActivities = buildRequest(
  "/decentralized/network/{network}",
  "get"
).withParams(
  (path) => ({
    params: objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getPlatformActivities = buildRequest(
  "/decentralized/platform/{platform}",
  "get"
).withParams(
  (path) => ({
    params: objectToSnake({ path })
  }),
  ({ data, meta }) => ({
    data,
    cursor: meta?.cursor
  })
);
const getFederatedActivitiesByAccount = buildRequest(
  "/federated/{account}",
  "get"
).withParams(({ account, ...query }) => ({
  params: objectToSnake({ path: { account }, query })
}));
const getFederatedActivity = buildRequest(
  "/federated/tx/{id}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getFederatedActivitiesByAccounts = buildRequest(
  "/federated/accounts",
  "post"
).withParams((body) => ({
  body: objectToSnake(body)
}));
const getFederatedNetworkActivities = buildRequest(
  "/federated/network/{network}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getFederatedPlatformActivities = buildRequest(
  "/federated/platform/{platform}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getBridgingTransactions = buildRequest(
  "/nta/bridgings/transactions",
  "get"
).withParams((query) => ({
  params: objectToSnake({ query })
}));
const getBridgingTransaction = buildRequest(
  "/nta/bridgings/transactions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getStakingTransactions = buildRequest(
  "/nta/stakings/transactions",
  "get"
).withParams((query) => ({
  params: objectToSnake({ query })
}));
const getStakingTransaction = buildRequest(
  "/nta/stakings/transactions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getStakings = buildRequest(
  "/nta/stakings/stakings",
  "get"
).withParams((query) => ({
  params: objectToSnake({ query })
}));
const getStakingProfit = buildRequest(
  "/nta/stakings/{staker_address}/profit",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getChips = buildRequest("/nta/chips", "get").withParams(
  (query) => ({
    params: objectToSnake({ query })
  })
);
const getChip = buildRequest("/nta/chips/{chip_id}", "get").withParams(
  (path) => ({
    params: objectToSnake({ path })
  })
);
const getChipSvgURL = (params, client = getDefaultClient()) => new URL(`/nta/chips/${params.chipId}/image.svg`, client.baseUrl);
const getChipSvgSrc = (params, client = getDefaultClient()) => getChipSvgURL(params, client).href;
const getChipSvg = buildRequest(
  "/nta/chips/{chip_id}/image.svg",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getNodeCountSnapshot = buildRequest(
  "/nta/snapshots/nodes/count",
  "get"
).withoutParams();
const getStakerCountSnapshot = buildRequest(
  "/nta/snapshots/stakers/count",
  "get"
).withoutParams();
const getStakerProfitSnapshot = buildRequest(
  "/nta/snapshots/stakers/profit",
  "get"
).withParams((query) => ({
  params: objectToSnake({ query })
}));
const getStakingStat = buildRequest(
  "/nta/stakings/{staker_address}/stat",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getOperationProfitSnapshot = buildRequest(
  "/nta/snapshots/nodes/operation/profit",
  "get"
).withParams((query) => ({
  params: objectToSnake({ query })
}));
const getEpochsApySnapshot = buildRequest(
  "/nta/snapshots/epochs/apy",
  "get"
).withoutParams();
const getAllNodes = buildRequest("/nta/nodes", "get").withParams(
  (query) => ({
    params: objectToSnake({ query })
  })
);
const getNode = buildRequest("/nta/nodes/{address}", "get").withParams(
  (path) => ({
    params: objectToSnake({ path })
  })
);
const getNodeAvatarSvgURL = (params, client = getDefaultClient()) => new URL(`/nta/nodes/${params.address}/avatar.svg`, client.baseUrl);
const getNodeAvatarSvgSrc = (params, client = getDefaultClient()) => getNodeAvatarSvgURL(params, client).href;
const getNodeAvatarSvg = buildRequest(
  "/nta/nodes/{address}/avatar.svg",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getNodeEvents = buildRequest(
  "/nta/nodes/{address}/events",
  "get"
).withParams(({ address, ...query }) => ({
  params: objectToSnake({ path: { address }, query })
}));
const getNodeOperationProfit = buildRequest(
  "/nta/nodes/{address}/operation/profit",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getEpochs = buildRequest("/nta/epochs", "get").withParams(
  (query) => ({
    params: objectToSnake({ query })
  })
);
const getEpoch = buildRequest(
  "/nta/epochs/{epoch_id}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getEpochTransaction = buildRequest(
  "/nta/epochs/distributions/{transaction_hash}",
  "get"
).withParams((path) => ({
  params: objectToSnake({ path })
}));
const getNodeRewards = buildRequest(
  "/nta/epochs/{address}/rewards",
  "get"
).withParams(({ address, ...query }) => ({
  params: objectToSnake({ path: { address }, query })
}));
const getAverageEpochsApy = buildRequest(
  "/nta/epochs/apy",
  "get"
).withoutParams();
const getNetworkAssets = buildRequest(
  "/nta/networks/assets",
  "get"
).withoutParams();
const getNetworkConfig = buildRequest(
  "/nta/networks/config",
  "get"
).withoutParams();
const getTotalRequests = buildRequest(
  "/nta/dsl/total_requests",
  "get"
).withoutParams();
const getTokenSupply = buildRequest(
  "/nta/token/supply",
  "get"
).withoutParams();
const getTokenTvl = buildRequest(
  "/nta/token/tvl",
  "get"
).withoutParams();

export { getActivities, getActivitiesByAccounts, getActivity, getAllNodes, getAverageEpochsApy, getBridgingTransaction, getBridgingTransactions, getChip, getChipSvg, getChipSvgSrc, getChipSvgURL, getChips, getEpoch, getEpochTransaction, getEpochs, getEpochsApySnapshot, getFederatedActivitiesByAccount, getFederatedActivitiesByAccounts, getFederatedActivity, getFederatedNetworkActivities, getFederatedPlatformActivities, getNetworkActivities, getNetworkAssets, getNetworkConfig, getNode, getNodeAvatarSvg, getNodeAvatarSvgSrc, getNodeAvatarSvgURL, getNodeCountSnapshot, getNodeEvents, getNodeOperationProfit, getNodeRewards, getOperationProfitSnapshot, getPlatformActivities, getRSSActivity, getStakerCountSnapshot, getStakerProfitSnapshot, getStakingProfit, getStakingStat, getStakingTransaction, getStakingTransactions, getStakings, getTokenSupply, getTokenTvl, getTotalRequests };
//# sourceMappingURL=index.js.map
//# sourceMappingURL=index.js.map