type ModuleName = string;
type ModulePath = string;
type ModuleData = unknown;
type ModuleIndex = number;

type ImportModulesMode = 'sync' | 'async';
type ImportModulesCallback = (name: ModuleName, path: ModulePath, data: ModuleData, index: ModuleIndex) => void;

interface ImportedModules {
  [key: ModulePath]: ModuleData;
}

interface ImportedModulesPublicOptions {
  includeSubdirectories?: boolean;
  targetDirectoryPath?: string;
  importPattern?: RegExp;
  importMode?: ImportModulesMode;
  limit?: number;
  forceReload?: boolean;
}

/**
 * Import modules from the current directory synchronously
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(): ImportedModules;
/**
 * Import modules from the current directory synchronously and call the provided callback for each imported module.
 * @param {ImportModulesCallback} callback - The callback function to call for each imported module.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(callback: ImportModulesCallback): ImportedModules;
/**
 * Import modules from the specified directory synchronously.
 * @param {string} targetDirectoryPath - The path to the directory to import modules from.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(targetDirectoryPath: string): ImportedModules;
/**
 * Import all modules from the specified directory synchronously and call the provided callback for each imported module.
 * @param {string} targetDirectoryPath - The path to the directory to import modules from.
 * @param {ImportModulesCallback} callback - The callback function to call for each imported module.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(targetDirectoryPath: string, callback: ImportModulesCallback): ImportedModules;
/**
 * Import all modules from the specified directory synchronously or asynchronously.
 * @param {string} targetDirectoryPath - The path to the directory to import modules from.
 * @param {ImportModulesMode} mode - The import mode. Can be 'sync' or 'async'.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(targetDirectoryPath: string, mode: ImportModulesMode): ImportedModules;
/**
 * Import all modules from the specified directory synchronously or asynchronously and call the provided callback for each imported module.
 * @param {string} targetDirectoryPath - The path to the directory to import modules from.
 * @param {ImportModulesMode} importMode - The import mode. Can be 'sync' or 'async'.
 * @param {ImportModulesCallback} callback - The callback function to call for each imported module.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(targetDirectoryPath: string, importMode: ImportModulesMode, callback: ImportModulesCallback): ImportedModules;
/**
 * Import all modules from the specified directory
 * @param {ImportedModulesPublicOptions} options - The options object.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(options: ImportedModulesPublicOptions): ImportedModules;
/**
 * Import all modules from the specified directory and call the provided callback for each imported module.
 * @param {ImportedModulesPublicOptions} options - The options object.
 * @param {ImportModulesCallback} callback - The callback function to call for each imported module.
 * @returns {ImportedModules} An object containing all imported modules.
 */
declare function directoryImport(options: ImportedModulesPublicOptions, callback: ImportModulesCallback): ImportedModules;

export { directoryImport };
