import { OpenAPIRegistry, OpenApiGeneratorV3, OpenApiGeneratorV31, extendZodWithOpenApi, getOpenApiMetadata } from "@asteasolutions/zod-to-openapi";
import { zValidator } from "@hono/zod-validator";
import { Hono } from "hono";
import { mergePath } from "hono/utils/url";
import { z } from "zod";

//#region src/zod-typeguard.ts
function isObject(x) {
	return typeof x === "object" && x !== null;
}
function isZod(x) {
	if (!x) return false;
	if (!isObject(x)) return false;
	return typeof x.parse === "function" && typeof x.safeParse === "function" && typeof x.parseAsync === "function" && typeof x.safeParseAsync === "function";
}

//#endregion
//#region src/index.ts
/**
* Converts a Hono instance to OpenAPIHono type.
* Use this function to restore the OpenAPIHono type after chaining methods like `get`, `post`, `use`, etc.
* @example
* ```ts
* import { OpenAPIHono, $ } from '@hono/zod-openapi'
*
* const app = $(
*   new OpenAPIHono().use(middleware)
* )
* app.openapi(route, handler)
* ```
*/
const $ = (app) => {
	return app;
};
var OpenAPIHono = class OpenAPIHono extends Hono {
	openAPIRegistry;
	defaultHook;
	constructor(init) {
		super(init);
		this.openAPIRegistry = new OpenAPIRegistry();
		this.defaultHook = init === null || init === void 0 ? void 0 : init.defaultHook;
	}
	/**
	*
	* @param {RouteConfig} route - The route definition which you create with `createRoute()`.
	* @param {Handler} handler - The handler. If you want to return a JSON object, you should specify the status code with `c.json()`.
	* @param {Hook} hook - Optional. The hook method defines what it should do after validation.
	* @example
	* app.openapi(
	*   route,
	*   (c) => {
	*     // ...
	*     return c.json(
	*       {
	*         age: 20,
	*         name: 'Young man',
	*       },
	*       200 // You should specify the status code even if it's 200.
	*     )
	*   },
	*  (result, c) => {
	*    if (!result.success) {
	*      return c.json(
	*        {
	*          code: 400,
	*          message: 'Custom Message',
	*        },
	*        400
	*      )
	*    }
	*  }
	*)
	*/
	openapi = ({ middleware: routeMiddleware, hide,...route }, handler, hook = this.defaultHook) => {
		var _route$request, _route$request2, _route$request3, _route$request4, _route$request5;
		if (!hide) this.openAPIRegistry.registerPath(route);
		const validators = [];
		if ((_route$request = route.request) === null || _route$request === void 0 ? void 0 : _route$request.query) {
			const validator = zValidator("query", route.request.query, hook);
			validators.push(validator);
		}
		if ((_route$request2 = route.request) === null || _route$request2 === void 0 ? void 0 : _route$request2.params) {
			const validator = zValidator("param", route.request.params, hook);
			validators.push(validator);
		}
		if ((_route$request3 = route.request) === null || _route$request3 === void 0 ? void 0 : _route$request3.headers) {
			const validator = zValidator("header", route.request.headers, hook);
			validators.push(validator);
		}
		if ((_route$request4 = route.request) === null || _route$request4 === void 0 ? void 0 : _route$request4.cookies) {
			const validator = zValidator("cookie", route.request.cookies, hook);
			validators.push(validator);
		}
		const bodyContent = (_route$request5 = route.request) === null || _route$request5 === void 0 || (_route$request5 = _route$request5.body) === null || _route$request5 === void 0 ? void 0 : _route$request5.content;
		if (bodyContent) for (const mediaType of Object.keys(bodyContent)) {
			if (!bodyContent[mediaType]) continue;
			const schema = bodyContent[mediaType]["schema"];
			if (!isZod(schema)) continue;
			if (isJSONContentType(mediaType)) {
				var _route$request6;
				const validator = zValidator("json", schema, hook);
				if ((_route$request6 = route.request) === null || _route$request6 === void 0 || (_route$request6 = _route$request6.body) === null || _route$request6 === void 0 ? void 0 : _route$request6.required) validators.push(validator);
				else {
					const mw = async (c, next) => {
						if (c.req.header("content-type")) {
							if (isJSONContentType(c.req.header("content-type"))) return await validator(c, next);
						}
						c.req.addValidatedData("json", {});
						await next();
					};
					validators.push(mw);
				}
			}
			if (isFormContentType(mediaType)) {
				var _route$request7;
				const validator = zValidator("form", schema, hook);
				if ((_route$request7 = route.request) === null || _route$request7 === void 0 || (_route$request7 = _route$request7.body) === null || _route$request7 === void 0 ? void 0 : _route$request7.required) validators.push(validator);
				else {
					const mw = async (c, next) => {
						if (c.req.header("content-type")) {
							if (isFormContentType(c.req.header("content-type"))) {
								await validator(c, next);
								return;
							}
						}
						c.req.addValidatedData("form", {});
						await next();
					};
					validators.push(mw);
				}
			}
		}
		const middleware = routeMiddleware ? Array.isArray(routeMiddleware) ? routeMiddleware : [routeMiddleware] : [];
		this.on([route.method], [route.path.replaceAll(/\/{(.+?)}/g, "/:$1")], ...middleware, ...validators, handler);
		return this;
	};
	getOpenAPIDocument = (objectConfig, generatorConfig) => {
		const document = new OpenApiGeneratorV3(this.openAPIRegistry.definitions, generatorConfig).generateDocument(objectConfig);
		return this._basePath ? addBasePathToDocument(document, this._basePath) : document;
	};
	getOpenAPI31Document = (objectConfig, generatorConfig) => {
		const document = new OpenApiGeneratorV31(this.openAPIRegistry.definitions, generatorConfig).generateDocument(objectConfig);
		return this._basePath ? addBasePathToDocument(document, this._basePath) : document;
	};
	doc = (path, configureObject, configureGenerator) => {
		return this.get(path, (c) => {
			const objectConfig = typeof configureObject === "function" ? configureObject(c) : configureObject;
			const generatorConfig = typeof configureGenerator === "function" ? configureGenerator(c) : configureGenerator;
			try {
				const document = this.getOpenAPIDocument(objectConfig, generatorConfig);
				return c.json(document);
			} catch (e) {
				return c.json(e, 500);
			}
		});
	};
	doc31 = (path, configureObject, configureGenerator) => {
		return this.get(path, (c) => {
			const objectConfig = typeof configureObject === "function" ? configureObject(c) : configureObject;
			const generatorConfig = typeof configureGenerator === "function" ? configureGenerator(c) : configureGenerator;
			try {
				const document = this.getOpenAPI31Document(objectConfig, generatorConfig);
				return c.json(document);
			} catch (e) {
				return c.json(e, 500);
			}
		});
	};
	route(path, app) {
		const pathForOpenAPI = path.replaceAll(/:([^\/]+)/g, "{$1}");
		super.route(path, app);
		if (!(app instanceof OpenAPIHono)) return this;
		app.openAPIRegistry.definitions.forEach((def) => {
			switch (def.type) {
				case "component": return this.openAPIRegistry.registerComponent(def.componentType, def.name, def.component);
				case "route":
					this.openAPIRegistry.registerPath({
						...def.route,
						path: mergePath(pathForOpenAPI, app._basePath.replaceAll(/:([^\/]+)/g, "{$1}"), def.route.path)
					});
					return;
				case "webhook":
					this.openAPIRegistry.registerWebhook({
						...def.webhook,
						path: mergePath(pathForOpenAPI, app._basePath.replaceAll(/:([^\/]+)/g, "{$1}"), def.webhook.path)
					});
					return;
				case "schema":
					var _getOpenApiMetadata$_;
					return this.openAPIRegistry.register((_getOpenApiMetadata$_ = getOpenApiMetadata(def.schema)._internal) === null || _getOpenApiMetadata$_ === void 0 ? void 0 : _getOpenApiMetadata$_.refId, def.schema);
				case "parameter":
					var _getOpenApiMetadata$_2;
					return this.openAPIRegistry.registerParameter((_getOpenApiMetadata$_2 = getOpenApiMetadata(def.schema)._internal) === null || _getOpenApiMetadata$_2 === void 0 ? void 0 : _getOpenApiMetadata$_2.refId, def.schema);
				default: {
					const errorIfNotExhaustive = def;
					throw new Error(`Unknown registry type: ${errorIfNotExhaustive}`);
				}
			}
		});
		return this;
	}
	basePath(path) {
		return new OpenAPIHono({
			...super.basePath(path),
			defaultHook: this.defaultHook
		});
	}
};
const createRoute = (routeConfig) => {
	const route = {
		...routeConfig,
		getRoutingPath() {
			return routeConfig.path.replaceAll(/\/{(.+?)}/g, "/:$1");
		}
	};
	return Object.defineProperty(route, "getRoutingPath", { enumerable: false });
};
extendZodWithOpenApi(z);
function addBasePathToDocument(document, basePath) {
	const updatedPaths = {};
	Object.keys(document.paths).forEach((path) => {
		updatedPaths[mergePath(basePath.replaceAll(/:([^\/]+)/g, "{$1}"), path)] = document.paths[path];
	});
	return {
		...document,
		paths: updatedPaths
	};
}
function isJSONContentType(contentType) {
	return /^application\/([a-z-\.]+\+)?json/.test(contentType);
}
function isFormContentType(contentType) {
	return contentType.startsWith("multipart/form-data") || contentType.startsWith("application/x-www-form-urlencoded");
}

//#endregion
export { $, OpenAPIHono, createRoute, extendZodWithOpenApi, z };
//# sourceMappingURL=index.js.map