import { Writable } from 'stream';
import { kWebSocket } from './constants';

interface ReceiverOptions {
    allowSynchronousEvents?: boolean;
    binaryType?: string;
    extensions?: Record<string, any>;
    isServer?: boolean;
    maxPayload?: number;
    skipUTF8Validation?: boolean;
}
/**
 * HyBi Receiver implementation.
 *
 * @extends Writable
 */
export declare class Receiver extends Writable {
    private _allowSynchronousEvents;
    private _binaryType;
    private _extensions;
    private _isServer;
    private _maxPayload;
    private _skipUTF8Validation;
    private [kWebSocket];
    private _bufferedBytes;
    private _buffers;
    private _compressed;
    private _payloadLength;
    private _mask;
    private _fragmented;
    private _masked;
    private _fin;
    private _opcode;
    private _totalPayloadLength;
    private _messageLength;
    private _fragments;
    private _errored;
    private _loop;
    private _state;
    constructor(options?: ReceiverOptions);
    /**
     * Implements `Writable.prototype._write()`.
     *
     * @param {Buffer} chunk The chunk of data to write
     * @param {string} encoding The character encoding of `chunk`
     * @param {Function} cb Callback
     * @private
     */
    _write(chunk: Buffer, encoding: string, cb: (err?: Error | null) => void): void;
    /**
     * Consumes `n` bytes from the buffered data.
     *
     * @param {number} n The number of bytes to consume
     * @return {Buffer} The consumed bytes
     * @private
     */
    consume(n: number): Buffer;
    /**
     * Starts the parsing loop.
     *
     * @param {Function} cb Callback
     * @private
     */
    startLoop(cb: (err?: Error | null) => void): void;
    /**
     * Reads the first two bytes of a frame.
     *
     * @param {Function} cb Callback
     * @private
     */
    getInfo(cb: (err?: Error | null) => void): void;
    /**
     * Reads the next two bytes and stores the message length.
     *
     * @param {Function} cb Callback
     * @private
     */
    getPayloadLength16(cb: (err?: Error | null) => void): void;
    /**
     * Reads the next eight bytes and stores the message length.
     *
     * @param {Function} cb Callback
     * @private
     */
    getPayloadLength64(cb: (err?: Error | null) => void): void;
    /**
     * Payload length has been read.
     *
     * @param {Function} cb Callback
     * @private
     */
    haveLength(cb: (err?: Error | null) => void): void;
    /**
     * Reads the mask bytes.
     *
     * @private
     */
    getMask(): void;
    /**
     * Reads data bytes.
     *
     * @param {Function} cb Callback
     * @private
     */
    getData(cb: (err?: Error | null) => void): void;
    /**
     * Decompresses data.
     *
     * @param {Buffer} data Compressed data
     * @param {Function} cb Callback
     * @private
     */
    decompress(data: Buffer, cb: (err?: Error | null) => void): void;
    /**
     * Handles a data message.
     *
     * @param {Function} cb Callback
     * @private
     */
    dataMessage(cb: (err?: Error | null) => void): void;
    /**
     * Handles a control message.
     *
     * @param {Buffer} data Data to handle
     * @return {(Error|RangeError|undefined)} A possible error
     * @private
     */
    controlMessage(data: Buffer, cb: (err?: Error | null) => void): void;
    createError(ErrorCtor: any, message: any, prefix: any, statusCode: any, errorCode: any): any;
}
export {};
//# sourceMappingURL=receiver.d.ts.map