# media.py
#
# Copyright (C) 2025 The Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

from __future__ import annotations

from typing import TYPE_CHECKING

from PIL import Image
from gi.repository import GLib, Gdk

from upscaler.logger import logger

if TYPE_CHECKING:
    from pathlib import Path

image_formats = ("image/png", "image/jpeg", "image/webp")
video_formats = ("video/mp4",)


def create_texture_from_img(img: Image.Image) -> Gdk.Texture:
    """Create a Gdk.Texture from a PIL.Image."""
    img = img.convert("RGBA")

    width, height = img.size

    texture: Gdk.Texture = Gdk.MemoryTexture.new(
        width,
        height,
        Gdk.MemoryFormat.R8G8B8A8,
        GLib.Bytes.new(img.tobytes()),
        # "Stride" is the amount of bytes per row in a given image.
        # The R8G8B8A8 format is 4 bytes long (each channel is a byte (8-bits)), so we multiply the width by 4 to obtain the stride.
        width * 4,
    )

    return texture


class MediaFile:
    """Base class representing a media file."""

    def __init__(self, original_path: Path, temporary_path: Path | None = None) -> None:
        """Initialize the base media file class."""
        self.original_path = original_path
        self.temporary_path = temporary_path

    def get_dimension(self) -> tuple[int, int]:
        """Get dimension of the media."""
        raise NotImplementedError

    def get_preferred_input_path(self) -> Path:
        """
        Get the preferred input path.

        Returns:
            @self.temporary_path if it exists, otherwise @self.original_path.
        """
        if self.temporary_path:
            return self.temporary_path
        return self.original_path

    def remove_temporary_path(self) -> None:
        """Remove temporary file's path."""
        if not self.temporary_path:
            raise TypeError(self.temporary_path)

        try:
            self.temporary_path.unlink()
            logger.info(f"“{self.temporary_path.name}” deleted")
        except (TypeError, OSError) as error:
            raise TypeError from error


class ImageFile(MediaFile):
    """Image file."""

    def get_dimension(self) -> tuple[int, int]:
        """Get dimension of the image."""
        width, height = Image.open(str(self.original_path)).size
        return (width, height)
