# queue_row.py: Manage queue system.
#
# Copyright (C) 2024 Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

import subprocess
from gettext import gettext as _
from typing import TYPE_CHECKING, Any

from gi.repository import Adw, GObject, Gtk

from upscaler.logger import logger
from upscaler.media import MediaFile

if TYPE_CHECKING:
    from pathlib import Path


@Gtk.Template.from_resource("/io/gitlab/theevilskeleton/Upscaler/queue-row.ui")
class QueueRow(Adw.PreferencesRow):
    """A custom `Gtk.ListBoxRow` widget containing important information about queuing and upscaling."""

    __gtype_name__ = "QueueRow"

    thumbnail: Adw.Clamp = Gtk.Template.Child()
    progressbar: Gtk.ProgressBar = Gtk.Template.Child()
    button_cancel: Gtk.Button = Gtk.Template.Child()
    button_copy: Gtk.Button = Gtk.Template.Child()
    button_save: Gtk.Button = Gtk.Template.Child()
    button_open: Gtk.Button = Gtk.Template.Child()

    def __init__(
        self,
        media_file: MediaFile,
        command: tuple[str, ...] | None = None,
        canceled: bool = False,
        **kwargs: Any,
    ) -> None:
        """Initialize queue row widget."""
        super().__init__(**kwargs)

        self.process: subprocess.Popen[Any]
        self.destination_path: Path

        self.media_file = media_file
        self.canceled = canceled
        self.command = command

        self.button_cancel.connect("clicked", self._on_cancel_clicked_cb)

    def run(self) -> None:
        """Run command if @self.command is set."""
        if self.command:
            self.button_cancel.set_tooltip_text(_("Stop"))
            self.process = subprocess.Popen(
                self.command,
                stderr=subprocess.PIPE,
                universal_newlines=True,
            )

    @GObject.Signal
    def aborted(self) -> None:
        """Emit the `aborted` signal."""
        logger.info("Signal 'abort' emitted")

    def _on_cancel_clicked_cb(self, *_args: Any) -> None:
        if hasattr(self, "process"):
            title = _("Stop Upscaling?")
            subtitle = _("All progress will be lost")
        else:
            title = _("Remove From Queue?")
            subtitle = _("Its order will be lost")

        dialog = Adw.AlertDialog.new(
            title,
            subtitle,
        )

        def response(_dialog: Adw.AlertDialog, response_id: str) -> None:
            if response_id == "stop":
                self.emit("aborted")

        dialog.add_response("cancel", _("_Cancel"))
        dialog.add_response("stop", _("_Stop"))
        dialog.set_response_appearance("stop", Adw.ResponseAppearance.DESTRUCTIVE)
        dialog.connect("response", response)
        dialog.present(self)
