# scale_comparison_frame.py
#
# Copyright (C) 2025 The Upscaler Contributors
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-only

from __future__ import annotations

from typing import TYPE_CHECKING, Any

from gi.repository import Adw, Gtk

# ruff: noqa: TC001
from upscaler.dimension_label import DimensionLabel

if TYPE_CHECKING:
    from gi.repository import Gdk

    from upscaler.media import MediaFile


@Gtk.Template.from_resource(
    "/io/gitlab/theevilskeleton/Upscaler/scale-comparison-frame.ui"
)
class ScaleComparisonFrame(Gtk.Widget):
    """Display the input and output dimensions, with a frame above."""

    __gtype_name__ = "ScaleComparisonFrame"

    image: Gtk.Picture = Gtk.Template.Child()
    input_dimension: DimensionLabel = Gtk.Template.Child()
    output_dimension: DimensionLabel = Gtk.Template.Child()
    remove_button: Gtk.Button = Gtk.Template.Child()

    def __init__(
        self,
        media_file: MediaFile,
        texture: Gdk.Texture,
        scale_factor: int,
        **kwargs: Any,
    ) -> None:
        """Initialize ScaleComparisonFrame."""
        super().__init__(css_name="scalecomparisonframe", **kwargs)

        self.animation = Adw.TimedAnimation.new(
            self, 0, 1, 200, Adw.PropertyAnimationTarget.new(self, "opacity")
        )
        self.animation.play()

        self.media_file = media_file
        self.image.set_paintable(texture)

        width, height = self.media_file.get_dimension()
        self.input_dimension.set_dimension_label(width, height)
        self.set_output_dimension_by_scale_factor(scale_factor)

        self.remove_button.connect("clicked", self._on_remove_clicked_cb)

    def _on_remove_clicked_cb(self, *_args: Any) -> None:
        def done_cb(*_args: Any) -> None:
            parent = self.get_parent()
            if isinstance(parent, Adw.Carousel):
                parent.remove(self)

        self.add_css_class("removing")
        self.animation.set_reverse(True)
        self.animation.connect("done", done_cb)
        self.animation.play()

    def set_output_dimension_by_scale_factor(self, scale_factor: int) -> None:
        """Set output dimension label using the given scale factor."""
        width, height = self.input_dimension.get_dimension_label()
        self.output_dimension.set_dimension_label(
            width * scale_factor,
            height * scale_factor,
        )
