﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/snowball/SnowballRequest.h>
#include <aws/snowball/Snowball_EXPORTS.h>
#include <aws/snowball/model/JobResource.h>
#include <aws/snowball/model/Notification.h>
#include <aws/snowball/model/OnDeviceServiceConfiguration.h>
#include <aws/snowball/model/ShippingOption.h>

#include <utility>

namespace Aws {
namespace Snowball {
namespace Model {

/**
 */
class UpdateClusterRequest : public SnowballRequest {
 public:
  AWS_SNOWBALL_API UpdateClusterRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateCluster"; }

  AWS_SNOWBALL_API Aws::String SerializePayload() const override;

  AWS_SNOWBALL_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The cluster ID of the cluster that you want to update, for example
   * <code>CID123e4567-e89b-12d3-a456-426655440000</code>.</p>
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  UpdateClusterRequest& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new role Amazon Resource Name (ARN) that you want to associate with this
   * cluster. To create a role ARN, use the <a
   * href="https://docs.aws.amazon.com/IAM/latest/APIReference/API_CreateRole.html">CreateRole</a>
   * API action in Identity and Access Management (IAM).</p>
   */
  inline const Aws::String& GetRoleARN() const { return m_roleARN; }
  inline bool RoleARNHasBeenSet() const { return m_roleARNHasBeenSet; }
  template <typename RoleARNT = Aws::String>
  void SetRoleARN(RoleARNT&& value) {
    m_roleARNHasBeenSet = true;
    m_roleARN = std::forward<RoleARNT>(value);
  }
  template <typename RoleARNT = Aws::String>
  UpdateClusterRequest& WithRoleARN(RoleARNT&& value) {
    SetRoleARN(std::forward<RoleARNT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated description of this cluster.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  UpdateClusterRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated arrays of <a>JobResource</a> objects that can include updated
   * <a>S3Resource</a> objects or <a>LambdaResource</a> objects.</p>
   */
  inline const JobResource& GetResources() const { return m_resources; }
  inline bool ResourcesHasBeenSet() const { return m_resourcesHasBeenSet; }
  template <typename ResourcesT = JobResource>
  void SetResources(ResourcesT&& value) {
    m_resourcesHasBeenSet = true;
    m_resources = std::forward<ResourcesT>(value);
  }
  template <typename ResourcesT = JobResource>
  UpdateClusterRequest& WithResources(ResourcesT&& value) {
    SetResources(std::forward<ResourcesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the service or services on the Snow Family device that your
   * transferred data will be exported from or imported into. Amazon Web Services
   * Snow Family device clusters support Amazon S3 and NFS (Network File System).</p>
   */
  inline const OnDeviceServiceConfiguration& GetOnDeviceServiceConfiguration() const { return m_onDeviceServiceConfiguration; }
  inline bool OnDeviceServiceConfigurationHasBeenSet() const { return m_onDeviceServiceConfigurationHasBeenSet; }
  template <typename OnDeviceServiceConfigurationT = OnDeviceServiceConfiguration>
  void SetOnDeviceServiceConfiguration(OnDeviceServiceConfigurationT&& value) {
    m_onDeviceServiceConfigurationHasBeenSet = true;
    m_onDeviceServiceConfiguration = std::forward<OnDeviceServiceConfigurationT>(value);
  }
  template <typename OnDeviceServiceConfigurationT = OnDeviceServiceConfiguration>
  UpdateClusterRequest& WithOnDeviceServiceConfiguration(OnDeviceServiceConfigurationT&& value) {
    SetOnDeviceServiceConfiguration(std::forward<OnDeviceServiceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the updated <a>Address</a> object.</p>
   */
  inline const Aws::String& GetAddressId() const { return m_addressId; }
  inline bool AddressIdHasBeenSet() const { return m_addressIdHasBeenSet; }
  template <typename AddressIdT = Aws::String>
  void SetAddressId(AddressIdT&& value) {
    m_addressIdHasBeenSet = true;
    m_addressId = std::forward<AddressIdT>(value);
  }
  template <typename AddressIdT = Aws::String>
  UpdateClusterRequest& WithAddressId(AddressIdT&& value) {
    SetAddressId(std::forward<AddressIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated shipping option value of this cluster's <a>ShippingDetails</a>
   * object.</p>
   */
  inline ShippingOption GetShippingOption() const { return m_shippingOption; }
  inline bool ShippingOptionHasBeenSet() const { return m_shippingOptionHasBeenSet; }
  inline void SetShippingOption(ShippingOption value) {
    m_shippingOptionHasBeenSet = true;
    m_shippingOption = value;
  }
  inline UpdateClusterRequest& WithShippingOption(ShippingOption value) {
    SetShippingOption(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The new or updated <a>Notification</a> object.</p>
   */
  inline const Notification& GetNotification() const { return m_notification; }
  inline bool NotificationHasBeenSet() const { return m_notificationHasBeenSet; }
  template <typename NotificationT = Notification>
  void SetNotification(NotificationT&& value) {
    m_notificationHasBeenSet = true;
    m_notification = std::forward<NotificationT>(value);
  }
  template <typename NotificationT = Notification>
  UpdateClusterRequest& WithNotification(NotificationT&& value) {
    SetNotification(std::forward<NotificationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The updated ID for the forwarding address for a cluster. This field is not
   * supported in most regions.</p>
   */
  inline const Aws::String& GetForwardingAddressId() const { return m_forwardingAddressId; }
  inline bool ForwardingAddressIdHasBeenSet() const { return m_forwardingAddressIdHasBeenSet; }
  template <typename ForwardingAddressIdT = Aws::String>
  void SetForwardingAddressId(ForwardingAddressIdT&& value) {
    m_forwardingAddressIdHasBeenSet = true;
    m_forwardingAddressId = std::forward<ForwardingAddressIdT>(value);
  }
  template <typename ForwardingAddressIdT = Aws::String>
  UpdateClusterRequest& WithForwardingAddressId(ForwardingAddressIdT&& value) {
    SetForwardingAddressId(std::forward<ForwardingAddressIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clusterId;

  Aws::String m_roleARN;

  Aws::String m_description;

  JobResource m_resources;

  OnDeviceServiceConfiguration m_onDeviceServiceConfiguration;

  Aws::String m_addressId;

  ShippingOption m_shippingOption{ShippingOption::NOT_SET};

  Notification m_notification;

  Aws::String m_forwardingAddressId;
  bool m_clusterIdHasBeenSet = false;
  bool m_roleARNHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_resourcesHasBeenSet = false;
  bool m_onDeviceServiceConfigurationHasBeenSet = false;
  bool m_addressIdHasBeenSet = false;
  bool m_shippingOptionHasBeenSet = false;
  bool m_notificationHasBeenSet = false;
  bool m_forwardingAddressIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Snowball
}  // namespace Aws
