﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iot-managed-integrations/IoTManagedIntegrationsRequest.h>
#include <aws/iot-managed-integrations/IoTManagedIntegrations_EXPORTS.h>
#include <aws/iot-managed-integrations/model/ProvisioningType.h>

#include <utility>

namespace Aws {
namespace IoTManagedIntegrations {
namespace Model {

/**
 */
class CreateProvisioningProfileRequest : public IoTManagedIntegrationsRequest {
 public:
  AWS_IOTMANAGEDINTEGRATIONS_API CreateProvisioningProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateProvisioningProfile"; }

  AWS_IOTMANAGEDINTEGRATIONS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The type of provisioning workflow the device uses for onboarding to IoT
   * managed integrations.</p>
   */
  inline ProvisioningType GetProvisioningType() const { return m_provisioningType; }
  inline bool ProvisioningTypeHasBeenSet() const { return m_provisioningTypeHasBeenSet; }
  inline void SetProvisioningType(ProvisioningType value) {
    m_provisioningTypeHasBeenSet = true;
    m_provisioningType = value;
  }
  inline CreateProvisioningProfileRequest& WithProvisioningType(ProvisioningType value) {
    SetProvisioningType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The id of the certificate authority (CA) certificate.</p>
   */
  inline const Aws::String& GetCaCertificate() const { return m_caCertificate; }
  inline bool CaCertificateHasBeenSet() const { return m_caCertificateHasBeenSet; }
  template <typename CaCertificateT = Aws::String>
  void SetCaCertificate(CaCertificateT&& value) {
    m_caCertificateHasBeenSet = true;
    m_caCertificate = std::forward<CaCertificateT>(value);
  }
  template <typename CaCertificateT = Aws::String>
  CreateProvisioningProfileRequest& WithCaCertificate(CaCertificateT&& value) {
    SetCaCertificate(std::forward<CaCertificateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The claim certificate.</p>
   */
  inline const Aws::String& GetClaimCertificate() const { return m_claimCertificate; }
  inline bool ClaimCertificateHasBeenSet() const { return m_claimCertificateHasBeenSet; }
  template <typename ClaimCertificateT = Aws::String>
  void SetClaimCertificate(ClaimCertificateT&& value) {
    m_claimCertificateHasBeenSet = true;
    m_claimCertificate = std::forward<ClaimCertificateT>(value);
  }
  template <typename ClaimCertificateT = Aws::String>
  CreateProvisioningProfileRequest& WithClaimCertificate(ClaimCertificateT&& value) {
    SetClaimCertificate(std::forward<ClaimCertificateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the provisioning template.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateProvisioningProfileRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An idempotency token. If you retry a request that completed successfully
   * initially using the same client token and parameters, then the retry attempt
   * will succeed without performing any further actions.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateProvisioningProfileRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A set of key/value pairs that are used to manage the provisioning
   * profile.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateProvisioningProfileRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateProvisioningProfileRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  ProvisioningType m_provisioningType{ProvisioningType::NOT_SET};

  Aws::String m_caCertificate;

  Aws::String m_claimCertificate;

  Aws::String m_name;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_provisioningTypeHasBeenSet = false;
  bool m_caCertificateHasBeenSet = false;
  bool m_claimCertificateHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace IoTManagedIntegrations
}  // namespace Aws
