/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import type { Config } from '../config/config.js';
import type { AgentDefinition, AgentInputs, OutputObject, SubagentActivityEvent } from './types.js';
/** A callback function to report on agent activity. */
export type ActivityCallback = (activity: SubagentActivityEvent) => void;
/**
 * Executes an agent loop based on an {@link AgentDefinition}.
 *
 * This executor uses a simplified two-phase approach:
 * 1.  **Work Phase:** The agent runs in a loop, calling tools until it has
 * gathered all necessary information to fulfill its goal.
 * 2.  **Extraction Phase:** A final prompt is sent to the model to summarize
 * the work and extract the final result in the desired format.
 */
export declare class AgentExecutor {
    readonly definition: AgentDefinition;
    private readonly agentId;
    private readonly toolRegistry;
    private readonly runtimeContext;
    private readonly onActivity?;
    /**
     * Creates and validates a new `AgentExecutor` instance.
     *
     * This method ensures that all tools specified in the agent's definition are
     * safe for non-interactive use before creating the executor.
     *
     * @param definition The definition object for the agent.
     * @param runtimeContext The global runtime configuration.
     * @param onActivity An optional callback to receive activity events.
     * @returns A promise that resolves to a new `AgentExecutor` instance.
     */
    static create(definition: AgentDefinition, runtimeContext: Config, onActivity?: ActivityCallback): Promise<AgentExecutor>;
    /**
     * Constructs a new AgentExecutor instance.
     *
     * @private This constructor is private. Use the static `create` method to
     * instantiate the class.
     */
    private constructor();
    /**
     * Runs the agent.
     *
     * @param inputs The validated input parameters for this invocation.
     * @param signal An `AbortSignal` for cancellation.
     * @returns A promise that resolves to the agent's final output.
     */
    run(inputs: AgentInputs, signal: AbortSignal): Promise<OutputObject>;
    /**
     * Calls the generative model with the current context and tools.
     *
     * @returns The model's response, including any tool calls or text.
     */
    private callModel;
    /** Initializes a `GeminiChat` instance for the agent run. */
    private createChatObject;
    /**
     * Executes function calls requested by the model and returns the results.
     *
     * @returns A new `Content` object to be added to the chat history.
     */
    private processFunctionCalls;
    /**
     * Prepares the list of tool function declarations to be sent to the model.
     */
    private prepareToolsList;
    /** Builds the system prompt from the agent definition and inputs. */
    private buildSystemPrompt;
    /** Builds the final message for the extraction phase. */
    private buildExtractionMessage;
    /**
     * Validates that all tools in a registry are safe for non-interactive use.
     *
     * @throws An error if a tool is not on the allow-list for non-interactive execution.
     */
    private static validateTools;
    /**
     * Checks if the agent should terminate due to exceeding configured limits.
     *
     * @returns The reason for termination, or `null` if execution can continue.
     */
    private checkTermination;
    /** Emits an activity event to the configured callback. */
    private emitActivity;
}
