﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/sesv2/SESV2Request.h>
#include <aws/sesv2/SESV2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SESV2 {
namespace Model {

/**
 * <p>Represents a request to associate a resource with a tenant.</p> <p>Resources
 * can be email identities, configuration sets, or email templates. When you
 * associate a resource with a tenant, you can use that resource when sending
 * emails on behalf of that tenant.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/sesv2-2019-09-27/CreateTenantResourceAssociationRequest">AWS
 * API Reference</a></p>
 */
class CreateTenantResourceAssociationRequest : public SESV2Request {
 public:
  AWS_SESV2_API CreateTenantResourceAssociationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateTenantResourceAssociation"; }

  AWS_SESV2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the tenant to associate the resource with.</p>
   */
  inline const Aws::String& GetTenantName() const { return m_tenantName; }
  inline bool TenantNameHasBeenSet() const { return m_tenantNameHasBeenSet; }
  template <typename TenantNameT = Aws::String>
  void SetTenantName(TenantNameT&& value) {
    m_tenantNameHasBeenSet = true;
    m_tenantName = std::forward<TenantNameT>(value);
  }
  template <typename TenantNameT = Aws::String>
  CreateTenantResourceAssociationRequest& WithTenantName(TenantNameT&& value) {
    SetTenantName(std::forward<TenantNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the resource to associate with the
   * tenant.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  CreateTenantResourceAssociationRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_tenantName;

  Aws::String m_resourceArn;
  bool m_tenantNameHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace SESV2
}  // namespace Aws
