import { app, BrowserWindow } from 'electron';
import { ELECTRON_MAJOR_VERSION } from '../electron-normalize.js';
import { startSession, endSession, endSessionOnExit } from '../sessions.js';

function focusedWindow() {
    for (const window of BrowserWindow.getAllWindows()) {
        if (!window.isDestroyed() && window.webContents && !window.webContents.isDestroyed()) {
            // It's important to test both isVisible and isFocused, since
            // Electron v12-13 do not report hidden as a loss of focus
            if (window.isFocused() && window.isVisible()) {
                return true;
            }
        }
    }
    return false;
}
/**
 * Tracks sessions as BrowserWindows focused.
 *
 * Supports Electron >= v12
 */
class BrowserWindowSession {
    constructor(_options = {}) {
        this._options = _options;
        this._windowStateChanged = () => {
            var _a;
            const hasFocusedWindow = focusedWindow();
            if (hasFocusedWindow) {
                // We are now active
                if (this._state.name === 'inactive') {
                    // If we were inactive, start a new session
                    void startSession(true);
                }
                else if (this._state.name === 'timeout') {
                    // Clear the timeout since the app has become active again
                    clearTimeout(this._state.timer);
                }
                this._state = { name: 'active' };
            }
            else {
                if (this._state.name === 'active') {
                    // We have become inactive, start the timeout
                    const timeout = ((_a = this._options.backgroundTimeoutSeconds) !== null && _a !== void 0 ? _a : 30) * 1000;
                    const timer = setTimeout(() => {
                        // if the state says we're still waiting for the timeout, end the session
                        if (this._state.name === 'timeout') {
                            this._state = { name: 'inactive' };
                            void endSession();
                        }
                    }, timeout)
                        // unref so this timer doesn't block app exit
                        .unref();
                    this._state = { name: 'timeout', timer };
                }
            }
        };
        if (ELECTRON_MAJOR_VERSION < 12) {
            throw new Error('BrowserWindowSession requires Electron >= v12');
        }
        this.name = BrowserWindowSession.id;
        this._state = { name: 'inactive' };
    }
    /** @inheritDoc */
    setupOnce() {
        app.on('browser-window-created', (_event, window) => {
            window.on('focus', this._windowStateChanged);
            window.on('blur', this._windowStateChanged);
            window.on('show', this._windowStateChanged);
            window.on('hide', this._windowStateChanged);
            // when the window is closed we need to remove the listeners
            window.once('closed', () => {
                window.removeListener('focus', this._windowStateChanged);
                window.removeListener('blur', this._windowStateChanged);
                window.removeListener('show', this._windowStateChanged);
                window.removeListener('hide', this._windowStateChanged);
            });
        });
        // if the app exits while the session is active, end the session
        endSessionOnExit();
    }
}
/** @inheritDoc */
BrowserWindowSession.id = 'BrowserWindowSession';

export { BrowserWindowSession };
//# sourceMappingURL=browser-window-session.js.map
