const tslib = require('tslib');
const core = require('@sentry/core');
const utils = require('@sentry/utils');
const electron = require('electron');
const context = require('../../context.js');
const electronNormalize = require('../../electron-normalize.js');
const fs = require('../../fs.js');
const renderers = require('../../renderers.js');
const sessions = require('../../sessions.js');
const store = require('../../store.js');
const minidumpLoader = require('./minidump-loader.js');
const merge = require('../../../common/merge.js');

/** Sends minidumps via the Sentry uploader */
class SentryMinidump {
    constructor() {
        this.name = SentryMinidump.id;
    }
    /** @inheritDoc */
    setupOnce() {
        // Mac AppStore builds cannot run the crash reporter due to the sandboxing
        // requirements. In this case, we prevent enabling native crashes entirely.
        // https://electronjs.org/docs/tutorial/mac-app-store-submission-guide#limitations-of-mas-build
        if (process.mas) {
            return;
        }
        this._startCrashReporter();
        this._scopeStore = new store.BufferedWriteStore(fs.sentryCachePath, 'scope_v3', {
            scope: new core.Scope(),
        });
        // We need to store the scope in a variable here so it can be attached to minidumps
        this._scopeLastRun = this._scopeStore.get();
        const hub = core.getCurrentHub();
        const client = hub.getClient();
        const options = client === null || client === void 0 ? void 0 : client.getOptions();
        const currentRelease = (options === null || options === void 0 ? void 0 : options.release) || context.getDefaultReleaseName();
        const currentEnvironment = (options === null || options === void 0 ? void 0 : options.environment) || context.getDefaultEnvironment();
        this._setupScopeListener(currentRelease, currentEnvironment);
        if (!(options === null || options === void 0 ? void 0 : options.dsn)) {
            throw new utils.SentryError('Attempted to enable Electron native crash reporter but no DSN was supplied');
        }
        renderers.trackRendererProperties();
        this._minidumpLoader = minidumpLoader.getMinidumpLoader();
        electronNormalize.onRendererProcessGone(electronNormalize.EXIT_REASONS, (contents, details) => this._sendRendererCrash(options, contents, details));
        electronNormalize.onChildProcessGone(electronNormalize.EXIT_REASONS, (details) => this._sendChildProcessCrash(options, details));
        // Start to submit recent minidump crashes. This will load breadcrumbs and
        // context information that was cached on disk prior to the crash.
        this._sendNativeCrashes({
            level: 'fatal',
            platform: 'native',
            tags: {
                'event.environment': 'native',
                'event.process': 'browser',
                event_type: 'native',
            },
        })
            .then((minidumpsFound) => 
        // Check for previous uncompleted session. If a previous session exists
        // and no minidumps were found, its likely an abnormal exit
        sessions.checkPreviousSession(minidumpsFound))
            .catch((error) => utils.logger.error(error));
    }
    /** Starts the native crash reporter */
    _startCrashReporter() {
        utils.logger.log('Starting Electron crashReporter');
        electron.crashReporter.start({
            companyName: '',
            ignoreSystemCrashHandler: true,
            productName: electron.app.name || electron.app.getName(),
            // Empty string doesn't work for Linux Crashpad and no submitURL doesn't work for older versions of Electron
            submitURL: 'https://f.a.k/e',
            uploadToServer: false,
            compress: true,
        });
    }
    /**
     * Helper function for sending renderer crashes
     */
    _sendRendererCrash(options, contents, details) {
        var _a;
        return tslib.__awaiter(this, void 0, void 0, function* () {
            const { getRendererName, release, environment } = options;
            const crashedProcess = (getRendererName === null || getRendererName === void 0 ? void 0 : getRendererName(contents)) || 'renderer';
            utils.logger.log(`'${crashedProcess}' process '${details.reason}'`);
            const event = merge.mergeEvents(yield context.getEventDefaults(release, environment), {
                contexts: {
                    electron: {
                        crashed_url: ((_a = renderers.getRendererProperties(contents.id)) === null || _a === void 0 ? void 0 : _a.url) || 'unknown',
                        details,
                    },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': crashedProcess,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield this._sendNativeCrashes(event);
            if (found) {
                sessions.sessionCrashed();
            }
        });
    }
    /**
     * Helper function for sending child process crashes
     */
    _sendChildProcessCrash(options, details) {
        return tslib.__awaiter(this, void 0, void 0, function* () {
            utils.logger.log(`${details.type} process has ${details.reason}`);
            const { release, environment } = options;
            const event = merge.mergeEvents(yield context.getEventDefaults(release, environment), {
                contexts: {
                    electron: { details },
                },
                level: 'fatal',
                // The default is javascript
                platform: 'native',
                tags: {
                    'event.environment': 'native',
                    'event.process': details.type,
                    'exit.reason': details.reason,
                    event_type: 'native',
                },
            });
            const found = yield this._sendNativeCrashes(event);
            if (found) {
                sessions.sessionCrashed();
            }
        });
    }
    /**
     * Adds a scope listener to persist changes to disk.
     */
    _setupScopeListener(currentRelease, currentEnvironment) {
        const scopeChanged = (updatedScope) => {
            const scope = core.Scope.clone(updatedScope);
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            scope._eventProcessors = [];
            // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access
            scope._scopeListeners = [];
            // Since the initial scope read is async, we need to ensure that any writes do not beat that
            // https://github.com/getsentry/sentry-electron/issues/585
            setImmediate(() => tslib.__awaiter(this, void 0, void 0, function* () {
                var _a;
                const event = yield context.getEventDefaults(currentRelease, currentEnvironment);
                void ((_a = this._scopeStore) === null || _a === void 0 ? void 0 : _a.set({
                    scope,
                    event,
                }));
            }));
        };
        const scope = core.getCurrentHub().getScope();
        if (scope) {
            scope.addScopeListener(scopeChanged);
            // Ensure at least one event is written to disk
            scopeChanged(scope);
        }
    }
    /**
     * Loads new native crashes from disk and sends them to Sentry.
     *
     * Returns true if one or more minidumps were found
     */
    _sendNativeCrashes(eventIn) {
        var _a, _b, _c, _d;
        return tslib.__awaiter(this, void 0, void 0, function* () {
            // Whenever we are called, assume that the crashes we are going to load down
            // below have occurred recently. This means, we can use the same event data
            // for all minidumps that we load now. There are two conditions:
            //
            //  1. The application crashed and we are just starting up. The stored
            //     breadcrumbs and context reflect the state during the application
            //     crash.
            //
            //  2. A renderer process crashed recently and we have just been notified
            //     about it. Just use the breadcrumbs and context information we have
            //     right now and hope that the delay was not too long.
            if (this._minidumpLoader === undefined) {
                throw new utils.SentryError('Invariant violation: Native crashes not enabled');
            }
            const hub = core.getCurrentHub();
            const client = hub.getClient();
            if (!client) {
                return true;
            }
            let event = eventIn;
            // If this is a native main process crash, we need to apply the scope and context from the previous run
            if (((_a = event.tags) === null || _a === void 0 ? void 0 : _a['event.process']) === 'browser') {
                const previousRun = yield this._scopeLastRun;
                const storedScope = core.Scope.clone(previousRun === null || previousRun === void 0 ? void 0 : previousRun.scope);
                event = yield storedScope.applyToEvent(event);
                if (event && previousRun) {
                    event.release = ((_b = previousRun.event) === null || _b === void 0 ? void 0 : _b.release) || event.release;
                    event.environment = ((_c = previousRun.event) === null || _c === void 0 ? void 0 : _c.environment) || event.environment;
                    event.contexts = ((_d = previousRun.event) === null || _d === void 0 ? void 0 : _d.contexts) || event.contexts;
                }
            }
            const hubScope = hub.getScope();
            event = hubScope && event ? yield hubScope.applyToEvent(event) : event;
            if (!event) {
                return false;
            }
            // If the SDK is not enabled, tell the loader to delete all minidumps
            const deleteAll = client.getOptions().enabled === false;
            let minidumpSent = false;
            yield this._minidumpLoader(deleteAll, (attachment) => {
                core.captureEvent(event, { attachments: [attachment] });
                minidumpSent = true;
            });
            // Unset to recover memory
            this._scopeLastRun = undefined;
            return minidumpSent;
        });
    }
}
/** @inheritDoc */
SentryMinidump.id = 'SentryMinidump';

exports.SentryMinidump = SentryMinidump;
//# sourceMappingURL=index.js.map
