import { __awaiter } from 'tslib';
import { logger } from '@sentry/utils';
import { join, dirname } from 'path';
import { Mutex } from '../common/mutex.js';
import { unlinkAsync, mkdirp, writeFileAsync, readFileAsync, statAsync } from './fs.js';

const dateFormat = /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.*\d{0,10}Z$/;
/** JSON revive function to enable de-serialization of Date objects */
function dateReviver(_, value) {
    if (typeof value === 'string' && dateFormat.test(value)) {
        return new Date(value);
    }
    return value;
}
/**
 * Stores data serialized to a JSON file.
 */
class Store {
    /**
     * Creates a new store.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     */
    constructor(path, id, initial) {
        this._lock = new Mutex();
        this._path = join(path, `${id}.json`);
        this._initial = initial;
    }
    /**
     * Updates data by replacing it with the given value.
     * @param data New data to replace the previous one.
     */
    set(data) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this._lock.runExclusive(() => __awaiter(this, void 0, void 0, function* () {
                this._data = data;
                try {
                    if (data === undefined) {
                        try {
                            yield unlinkAsync(this._path);
                        }
                        catch (_) {
                            //
                        }
                    }
                    else {
                        yield mkdirp(dirname(this._path));
                        yield writeFileAsync(this._path, JSON.stringify(data));
                    }
                }
                catch (e) {
                    logger.warn('Failed to write to store', e);
                    // This usually fails due to anti virus scanners, issues in the file
                    // system, or problems with network drives. We cannot fix or handle this
                    // issue and must resume gracefully. Thus, we have to ignore this error.
                }
            }));
        });
    }
    /**
     * Returns the current data.
     *
     * When invoked for the first time, it will try to load previously stored data
     * from disk. If the file does not exist, the initial value provided to the
     * constructor is used.
     */
    get() {
        return __awaiter(this, void 0, void 0, function* () {
            return this._lock.runExclusive(() => __awaiter(this, void 0, void 0, function* () {
                if (this._data === undefined) {
                    try {
                        this._data = JSON.parse(yield readFileAsync(this._path, 'utf8'), dateReviver);
                    }
                    catch (e) {
                        this._data = this._initial;
                    }
                }
                return this._data;
            }));
        });
    }
    /**
     * Updates data by passing it through the given function.
     * @param fn A function receiving the current data and returning new one.
     */
    update(fn) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.set(fn(yield this.get()));
        });
    }
    /** Returns store to its initial state */
    clear() {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.set(this._initial);
        });
    }
    /** Gets the Date that the file was last modified */
    getModifiedDate() {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            try {
                return (_a = (yield statAsync(this._path))) === null || _a === void 0 ? void 0 : _a.mtime;
            }
            catch (_) {
                return undefined;
            }
        });
    }
}
/**
 * Extends Store to throttle writes.
 */
class BufferedWriteStore extends Store {
    /**
     * Creates a new ThrottledStore.
     *
     * @param path A unique filename to store this data.
     * @param id A unique filename to store this data.
     * @param initial An initial value to initialize data with.
     * @param throttleTime The minimum time between writes
     */
    constructor(path, id, initial, _throttleTime = 500) {
        super(path, id, initial);
        this._throttleTime = _throttleTime;
    }
    /** @inheritdoc */
    set(data) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            this._data = data;
            this._pendingWrite = {
                // We overwrite the data for the pending write so that the latest data is written in the next flush
                data,
                // If there is already a pending timeout, we keep that rather than starting the timeout again
                timeout: ((_a = this._pendingWrite) === null || _a === void 0 ? void 0 : _a.timeout) || setTimeout(() => this._writePending(), this._throttleTime),
            };
        });
    }
    /** Writes the pending write to disk */
    _writePending() {
        if (this._pendingWrite) {
            const data = this._pendingWrite.data;
            // Clear the pending write immediately so that subsequent writes can be queued
            this._pendingWrite = undefined;
            void super.set(data);
        }
    }
}

export { BufferedWriteStore, Store };
//# sourceMappingURL=store.js.map
