using LeastSquaresOptim, Statistics, Test

# StRD tests (in progress)
# http://www.itl.nist.gov/div898/strd/nls/nls_main.shtml

function misra1a()
    name = "misra1a"
    data = [
    10.07E0      77.6E0;
    14.73E0     114.9E0;
    17.94E0     141.1E0;
    23.93E0     190.8E0;
    29.61E0     239.9E0;
    35.18E0     289.0E0;
    40.02E0     332.8E0;
    44.82E0     378.4E0;
    50.76E0     434.8E0;
    55.05E0     477.3E0;
    61.01E0     536.8E0;
    66.40E0     593.1E0;
    75.47E0     689.1E0;
    81.78E0     760.0E0;
    ]
    parameters =[
    5.0000000000E+02        2.5000000000E+02        2.3894212918E+02;
    1.0000000000E-04        5.0000000000E-04        5.5015643181E-04;
    ]
    f(x, beta) = beta[1]*(1-exp(-beta[2]*x))
    solution = [2.3894212918E+02, 5.5015643181E-04]

    name, data, parameters, f, solution
end

function Chwirut2()
    name = "Chwirut2"
    data = [
    92.9000E0     0.500E0;
    57.1000E0     1.000E0;
    31.0500E0     1.750E0;
    11.5875E0     3.750E0;
    8.0250E0     5.750E0;
    63.6000E0     0.875E0;
    21.4000E0     2.250E0;
    14.2500E0     3.250E0;
    8.4750E0     5.250E0;
    63.8000E0     0.750E0;
    26.8000E0     1.750E0;
    16.4625E0     2.750E0;
    7.1250E0     4.750E0;
    67.3000E0     0.625E0;
    41.0000E0     1.250E0;
    21.1500E0     2.250E0;
    8.1750E0     4.250E0;
    81.5000E0      .500E0;
    13.1200E0     3.000E0;
    59.9000E0      .750E0;
    14.6200E0     3.000E0;
    32.9000E0     1.500E0;
    5.4400E0     6.000E0;
    12.5600E0     3.000E0;
    5.4400E0     6.000E0;
    32.0000E0     1.500E0;
    13.9500E0     3.000E0;
    75.8000E0      .500E0;
    20.0000E0     2.000E0;
    10.4200E0     4.000E0;
    59.5000E0      .750E0;
    21.6700E0     2.000E0;
    8.5500E0     5.000E0;
    62.0000E0      .750E0;
    20.2000E0     2.250E0;
    7.7600E0     3.750E0;
    3.7500E0     5.750E0;
    11.8100E0     3.000E0;
    54.7000E0      .750E0;
    23.7000E0     2.500E0;
    11.5500E0     4.000E0;
    61.3000E0      .750E0;
    17.7000E0     2.500E0;
    8.7400E0     4.000E0;
    59.2000E0      .750E0;
    16.3000E0     2.500E0;
    8.6200E0     4.000E0;
    81.0000E0      .500E0;
    4.8700E0     6.000E0;
    14.6200E0     3.000E0;
    81.7000E0      .500E0;
    17.1700E0     2.750E0;
    81.3000E0      .500E0;
    28.9000E0     1.750E0;
    ]

    parameters = [
    0.1 0.15;        
    0.01        0.008;     
    0.02        0.010;
    ]
    f(x, beta) = exp(-beta[1]*x)/(beta[2]+beta[3]*x)   
    solution =[ 1.6657666537E-01, 5.1653291286E-03, 1.2150007096E-02]

    name, data, parameters, f, solution
end

function Chwitrut1()
    name = "Chwirut1"
    data =[
    92.9000E0     0.5000E0;
    78.7000E0     0.6250E0;
    64.2000E0     0.7500E0;
    64.9000E0     0.8750E0;
    57.1000E0     1.0000E0;
    43.3000E0     1.2500E0;
    31.1000E0     1.7500E0;
    23.6000E0     2.2500E0;
    31.0500E0     1.7500E0;
    23.7750E0     2.2500E0;
    17.7375E0     2.7500E0;
    13.8000E0     3.2500E0;
    11.5875E0     3.7500E0;
    9.4125E0     4.2500E0;
    7.7250E0     4.7500E0;
    7.3500E0     5.2500E0;
    8.0250E0     5.7500E0;
    90.6000E0     0.5000E0;
    76.9000E0     0.6250E0;
    71.6000E0     0.7500E0;
    63.6000E0     0.8750E0;
    54.0000E0     1.0000E0;
    39.2000E0     1.2500E0;
    29.3000E0     1.7500E0;
    21.4000E0     2.2500E0;
    29.1750E0     1.7500E0;
    22.1250E0     2.2500E0;
    17.5125E0     2.7500E0;
    14.2500E0     3.2500E0;
    9.4500E0     3.7500E0;
    9.1500E0     4.2500E0;
    7.9125E0     4.7500E0;
    8.4750E0     5.2500E0;
    6.1125E0     5.7500E0;
    80.0000E0     0.5000E0;
    79.0000E0     0.6250E0;
    63.8000E0     0.7500E0;
    57.2000E0     0.8750E0;
    53.2000E0     1.0000E0;
    42.5000E0     1.2500E0;
    26.8000E0     1.7500E0;
    20.4000E0     2.2500E0;
    26.8500E0     1.7500E0;
    21.0000E0     2.2500E0;
    16.4625E0     2.7500E0;
    12.5250E0     3.2500E0;
    10.5375E0     3.7500E0;
    8.5875E0     4.2500E0;
    7.1250E0     4.7500E0;
    6.1125E0     5.2500E0;
    5.9625E0     5.7500E0;
    74.1000E0     0.5000E0;
    67.3000E0     0.6250E0;
    60.8000E0     0.7500E0;
    55.5000E0     0.8750E0;
    50.3000E0     1.0000E0;
    41.0000E0     1.2500E0;
    29.4000E0     1.7500E0;
    20.4000E0     2.2500E0;
    29.3625E0     1.7500E0;
    21.1500E0     2.2500E0;
    16.7625E0     2.7500E0;
    13.2000E0     3.2500E0;
    10.8750E0     3.7500E0;
    8.1750E0     4.2500E0;
    7.3500E0     4.7500E0;
    5.9625E0     5.2500E0;
    5.6250E0     5.7500E0;
    81.5000E0      .5000E0;
    62.4000E0      .7500E0;
    32.5000E0     1.5000E0;
    12.4100E0     3.0000E0;
    13.1200E0     3.0000E0;
    15.5600E0     3.0000E0;
    5.6300E0     6.0000E0;
    78.0000E0      .5000E0;
    59.9000E0      .7500E0;
    33.2000E0     1.5000E0;
    13.8400E0     3.0000E0;
    12.7500E0     3.0000E0;
    14.6200E0     3.0000E0;
    3.9400E0     6.0000E0;
    76.8000E0      .5000E0;
    61.0000E0      .7500E0;
    32.9000E0     1.5000E0;
    13.8700E0     3.0000E0;
    11.8100E0     3.0000E0;
    13.3100E0     3.0000E0;
    5.4400E0     6.0000E0;
    78.0000E0      .5000E0;
    63.5000E0      .7500E0;
    33.8000E0     1.5000E0;
    12.5600E0     3.0000E0;
    5.6300E0     6.0000E0;
    12.7500E0     3.0000E0;
    13.1200E0     3.0000E0;
    5.4400E0     6.0000E0;
    76.8000E0      .5000E0;
    60.0000E0      .7500E0;
    47.8000E0     1.0000E0;
    32.0000E0     1.5000E0;
    22.2000E0     2.0000E0;
    22.5700E0     2.0000E0;
    18.8200E0     2.5000E0;
    13.9500E0     3.0000E0;
    11.2500E0     4.0000E0;
    9.0000E0     5.0000E0;
    6.6700E0     6.0000E0;
    75.8000E0      .5000E0;
    62.0000E0      .7500E0;
    48.8000E0     1.0000E0;
    35.2000E0     1.5000E0;
    20.0000E0     2.0000E0;
    20.3200E0     2.0000E0;
    19.3100E0     2.5000E0;
    12.7500E0     3.0000E0;
    10.4200E0     4.0000E0;
    7.3100E0     5.0000E0;
    7.4200E0     6.0000E0;
    70.5000E0      .5000E0;
    59.5000E0      .7500E0;
    48.5000E0     1.0000E0;
    35.8000E0     1.5000E0;
    21.0000E0     2.0000E0;
    21.6700E0     2.0000E0;
    21.0000E0     2.5000E0;
    15.6400E0     3.0000E0;
    8.1700E0     4.0000E0;
    8.5500E0     5.0000E0;
    10.1200E0     6.0000E0;
    78.0000E0      .5000E0;
    66.0000E0      .6250E0;
    62.0000E0      .7500E0;
    58.0000E0      .8750E0;
    47.7000E0     1.0000E0;
    37.8000E0     1.2500E0;
    20.2000E0     2.2500E0;
    21.0700E0     2.2500E0;
    13.8700E0     2.7500E0;
    9.6700E0     3.2500E0;
    7.7600E0     3.7500E0;
    5.4400E0     4.2500E0;
    4.8700E0     4.7500E0;
    4.0100E0     5.2500E0;
    3.7500E0     5.7500E0;
    24.1900E0     3.0000E0;
    25.7600E0     3.0000E0;
    18.0700E0     3.0000E0;
    11.8100E0     3.0000E0;
    12.0700E0     3.0000E0;
    16.1200E0     3.0000E0;
    70.8000E0      .5000E0;
    54.7000E0      .7500E0;
    48.0000E0     1.0000E0;
    39.8000E0     1.5000E0;
    29.8000E0     2.0000E0;
    23.7000E0     2.5000E0;
    29.6200E0     2.0000E0;
    23.8100E0     2.5000E0;
    17.7000E0     3.0000E0;
    11.5500E0     4.0000E0;
    12.0700E0     5.0000E0;
    8.7400E0     6.0000E0;
    80.7000E0      .5000E0;
    61.3000E0      .7500E0;
    47.5000E0     1.0000E0;
    29.0000E0     1.5000E0;
    24.0000E0     2.0000E0;
    17.7000E0     2.5000E0;
    24.5600E0     2.0000E0;
    18.6700E0     2.5000E0;
    16.2400E0     3.0000E0;
    8.7400E0     4.0000E0;
    7.8700E0     5.0000E0;
    8.5100E0     6.0000E0;
    66.7000E0      .5000E0;
    59.2000E0      .7500E0;
    40.8000E0     1.0000E0;
    30.7000E0     1.5000E0;
    25.7000E0     2.0000E0;
    16.3000E0     2.5000E0;
    25.9900E0     2.0000E0;
    16.9500E0     2.5000E0;
    13.3500E0     3.0000E0;
    8.6200E0     4.0000E0;
    7.2000E0     5.0000E0;
    6.6400E0     6.0000E0;
    13.6900E0     3.0000E0;
    81.0000E0      .5000E0;
    64.5000E0      .7500E0;
    35.5000E0     1.5000E0;
    13.3100E0     3.0000E0;
    4.8700E0     6.0000E0;
    12.9400E0     3.0000E0;
    5.0600E0     6.0000E0;
    15.1900E0     3.0000E0;
    14.6200E0     3.0000E0;
    15.6400E0     3.0000E0;
    25.5000E0     1.7500E0;
    25.9500E0     1.7500E0;
    81.7000E0      .5000E0;
    61.6000E0      .7500E0;
    29.8000E0     1.7500E0;
    29.8100E0     1.7500E0;
    17.1700E0     2.7500E0;
    10.3900E0     3.7500E0;
    28.4000E0     1.7500E0;
    28.6900E0     1.7500E0;
    81.3000E0      .5000E0;
    60.9000E0      .7500E0;
    16.6500E0     2.7500E0;
    10.0500E0     3.7500E0;
    28.9000E0     1.7500E0;
    28.9500E0     1.7500E0]

    parameters = [
    0.1         0.15;       
    0.01        0.008;
    0.02        0.010;
    ] 
    f(x, beta) = exp(-beta[1]*x)/(beta[2]+beta[3]*x)   
    solution = [1.9027818370E-01, 6.1314004477E-03, 1.0530908399E-02]

    name, data, parameters, f, solution
end

function Lanczos3()
    name = "Lanczos3"
    data = [
    2.5134E+00  0.00000E+00;
    2.0443E+00  5.00000E-02;
    1.6684E+00  1.00000E-01;
    1.3664E+00  1.50000E-01;
    1.1232E+00  2.00000E-01;
    0.9269E+00  2.50000E-01;
    0.7679E+00  3.00000E-01;
    0.6389E+00  3.50000E-01;
    0.5338E+00  4.00000E-01;
    0.4479E+00  4.50000E-01;
    0.3776E+00  5.00000E-01;
    0.3197E+00  5.50000E-01;
    0.2720E+00  6.00000E-01;
    0.2325E+00  6.50000E-01;
    0.1997E+00  7.00000E-01;
    0.1723E+00  7.50000E-01;
    0.1493E+00  8.00000E-01;
    0.1301E+00  8.50000E-01;
    0.1138E+00  9.00000E-01;
    0.1000E+00  9.50000E-01;
    0.0883E+00  1.00000E+00;
    0.0783E+00  1.05000E+00;
    0.0698E+00  1.10000E+00;
    0.0624E+00  1.15000E+00]

    parameters = [       
    1.2         0.5;     
    0.3         0.7;     
    5.6         3.6;     
    5.5         4.2;     
    6.5         4  ;     
    7.6         6.3;     
    ]
    f(x, beta) = beta[1] * exp(-beta[2] * x) + beta[3] * exp(-beta[4]*x) + beta[5] * exp(-beta[6]*x)
    solution = [
 8.6816414977E-02,
 9.5498101505E-01,
 8.4400777463E-01,
 2.9515951832E+00,
 1.5825685901E+00,
 4.9863565084E+00
 ]

    name, data, parameters, f, solution
end

function Gauss1()
    name = "Gauss1"
    data = [
    97.62227    1.000000;
    97.80724    2.000000;
    96.62247    3.000000;
    92.59022    4.000000;
    91.23869    5.000000;
    95.32704    6.000000;
    90.35040    7.000000;
    89.46235    8.000000;
    91.72520    9.000000;
    89.86916   10.000000;
    86.88076    11.00000;
    85.94360    12.00000;
    87.60686    13.00000;
    86.25839    14.00000;
    80.74976    15.00000;
    83.03551    16.00000;
    88.25837    17.00000;
    82.01316    18.00000;
    82.74098    19.00000;
    83.30034    20.00000;
    81.27850    21.00000;
    81.85506    22.00000;
    80.75195    23.00000;
    80.09573    24.00000;
    81.07633    25.00000;
    78.81542    26.00000;
    78.38596    27.00000;
    79.93386    28.00000;
    79.48474    29.00000;
    79.95942    30.00000;
    76.10691    31.00000;
    78.39830    32.00000;
    81.43060    33.00000;
    82.48867    34.00000;
    81.65462    35.00000;
    80.84323    36.00000;
    88.68663    37.00000;
    84.74438    38.00000;
    86.83934    39.00000;
    85.97739    40.00000;
    91.28509    41.00000;
    97.22411    42.00000;
    93.51733    43.00000;
    94.10159    44.00000;
    101.91760    45.00000;
    98.43134    46.00000;
    110.4214    47.00000;
    107.6628    48.00000;
    111.7288    49.00000;
    116.5115    50.00000;
    120.7609    51.00000;
    123.9553    52.00000;
    124.2437    53.00000;
    130.7996    54.00000;
    133.2960    55.00000;
    130.7788    56.00000;
    132.0565    57.00000;
    138.6584    58.00000;
    142.9252    59.00000;
    142.7215    60.00000;
    144.1249    61.00000;
    147.4377    62.00000;
    148.2647    63.00000;
    152.0519    64.00000;
    147.3863    65.00000;
    149.2074    66.00000;
    148.9537    67.00000;
    144.5876    68.00000;
    148.1226    69.00000;
    148.0144    70.00000;
    143.8893    71.00000;
    140.9088    72.00000;
    143.4434    73.00000;
    139.3938    74.00000;
    135.9878    75.00000;
    136.3927    76.00000;
    126.7262    77.00000;
    124.4487    78.00000;
    122.8647    79.00000;
    113.8557    80.00000;
    113.7037    81.00000;
    106.8407    82.00000;
    107.0034    83.00000;
    102.46290    84.00000;
    96.09296    85.00000;
    94.57555    86.00000;
    86.98824    87.00000;
    84.90154    88.00000;
    81.18023    89.00000;
    76.40117    90.00000;
    67.09200    91.00000;
    72.67155    92.00000;
    68.10848    93.00000;
    67.99088    94.00000;
    63.34094    95.00000;
    60.55253    96.00000;
    56.18687    97.00000;
    53.64482    98.00000;
    53.70307    99.00000;
    48.07893   100.00000;
    42.21258   101.00000;
    45.65181   102.00000;
    41.69728   103.00000;
    41.24946   104.00000;
    39.21349   105.00000;
    37.71696    106.0000;
    36.68395    107.0000;
    37.30393    108.0000;
    37.43277    109.0000;
    37.45012    110.0000;
    32.64648    111.0000;
    31.84347    112.0000;
    31.39951    113.0000;
    26.68912    114.0000;
    32.25323    115.0000;
    27.61008    116.0000;
    33.58649    117.0000;
    28.10714    118.0000;
    30.26428    119.0000;
    28.01648    120.0000;
    29.11021    121.0000;
    23.02099    122.0000;
    25.65091    123.0000;
    28.50295    124.0000;
    25.23701    125.0000;
    26.13828    126.0000;
    33.53260    127.0000;
    29.25195    128.0000;
    27.09847    129.0000;
    26.52999    130.0000;
    25.52401    131.0000;
    26.69218    132.0000;
    24.55269    133.0000;
    27.71763    134.0000;
    25.20297    135.0000;
    25.61483    136.0000;
    25.06893    137.0000;
    27.63930    138.0000;
    24.94851    139.0000;
    25.86806    140.0000;
    22.48183    141.0000;
    26.90045    142.0000;
    25.39919    143.0000;
    17.90614    144.0000;
    23.76039    145.0000;
    25.89689    146.0000;
    27.64231    147.0000;
    22.86101    148.0000;
    26.47003    149.0000;
    23.72888    150.0000;
    27.54334    151.0000;
    30.52683    152.0000;
    28.07261    153.0000;
    34.92815    154.0000;
    28.29194    155.0000;
    34.19161    156.0000;
    35.41207    157.0000;
    37.09336    158.0000;
    40.98330    159.0000;
    39.53923    160.0000;
    47.80123    161.0000;
    47.46305    162.0000;
    51.04166    163.0000;
    54.58065    164.0000;
    57.53001    165.0000;
    61.42089    166.0000;
    62.79032    167.0000;
    68.51455    168.0000;
    70.23053    169.0000;
    74.42776    170.0000;
    76.59911    171.0000;
    81.62053    172.0000;
    83.42208    173.0000;
    79.17451    174.0000;
    88.56985    175.0000;
    85.66525    176.0000;
    86.55502    177.0000;
    90.65907    178.0000;
    84.27290    179.0000;
    85.72220    180.0000;
    83.10702    181.0000;
    82.16884    182.0000;
    80.42568    183.0000;
    78.15692    184.0000;
    79.79691    185.0000;
    77.84378    186.0000;
    74.50327    187.0000;
    71.57289    188.0000;
    65.88031    189.0000;
    65.01385    190.0000;
    60.19582    191.0000;
    59.66726    192.0000;
    52.95478    193.0000;
    53.87792    194.0000;
    44.91274    195.0000;
    41.09909    196.0000;
    41.68018    197.0000;
    34.53379    198.0000;
    34.86419    199.0000;
    33.14787    200.0000;
    29.58864    201.0000;
    27.29462    202.0000;
    21.91439    203.0000;
    19.08159    204.0000;
    24.90290    205.0000;
    19.82341    206.0000;
    16.75551    207.0000;
    18.24558    208.0000;
    17.23549    209.0000;
    16.34934    210.0000;
    13.71285    211.0000;
    14.75676    212.0000;
    13.97169    213.0000;
    12.42867    214.0000;
    14.35519    215.0000;
    7.703309    216.0000;
    10.234410    217.0000;
    11.78315    218.0000;
    13.87768    219.0000;
    4.535700    220.0000;
    10.059280    221.0000;
    8.424824    222.0000;
    10.533120    223.0000;
    9.602255    224.0000;
    7.877514    225.0000;
    6.258121    226.0000;
    8.899865    227.0000;
    7.877754    228.0000;
    12.51191    229.0000;
    10.66205    230.0000;
    6.035400    231.0000;
    6.790655    232.0000;
    8.783535    233.0000;
    4.600288    234.0000;
    8.400915    235.0000;
    7.216561    236.0000;
    10.017410    237.0000;
    7.331278    238.0000;
    6.527863    239.0000;
    2.842001    240.0000;
    10.325070    241.0000;
    4.790995    242.0000;
    8.377101    243.0000;
    6.264445    244.0000;
    2.706213    245.0000;
    8.362329    246.0000;
    8.983658    247.0000;
    3.362571    248.0000;
    1.182746    249.0000;
    4.875359    250.0000;
    ]

    parameters = [
    97.0        94.0;
    0.009       0.0105;
    100.0        99.0;
    65.0        63.0;
    20.0        25.0;
    70.0        71.0 ;
    178.0       180.0;
    16.5        20.0;
    ]
    
    solution = [
    9.8778210871E+01,
    1.0497276517E-02,
    1.0048990633E+02,
    6.7481111276E+01,
    2.3129773360E+01,
    7.1994503004E+01,
    1.7899805021E+02,
    1.8389389025E+01
    ]
    f(x, beta) = beta[1] * exp(-beta[2]x) + beta[3]*exp(-(x - beta[4])^2 / beta[5]^2) + beta[6]*exp(-(x - beta[7])^2 / beta[8]^2)

    return name, data, parameters, f, solution
end

function Gauss2()
    name = "Gauss2"
    data = [
    97.58776    1.000000;
    97.76344    2.000000;
    96.56705    3.000000;
    92.52037    4.000000;
    91.15097    5.000000;
    95.21728    6.000000;
    90.21355    7.000000;
    89.29235    8.000000;
    91.51479    9.000000;
    89.60966   10.000000;
    86.56187    11.00000;
    85.55316    12.00000;
    87.13054    13.00000;
    85.67940    14.00000;
    80.04851    15.00000;
    82.18925    16.00000;
    87.24081    17.00000;
    80.79407    18.00000;
    81.28570    19.00000;
    81.56940    20.00000;
    79.22715    21.00000;
    79.43275    22.00000;
    77.90195    23.00000;
    76.75468    24.00000;
    77.17377    25.00000;
    74.27348    26.00000;
    73.11900    27.00000;
    73.84826    28.00000;
    72.47870    29.00000;
    71.92292    30.00000;
    66.92176    31.00000;
    67.93835    32.00000;
    69.56207    33.00000;
    69.07066    34.00000;
    66.53983    35.00000;
    63.87883    36.00000;
    69.71537    37.00000;
    63.60588    38.00000;
    63.37154    39.00000;
    60.01835    40.00000;
    62.67481    41.00000;
    65.80666    42.00000;
    59.14304    43.00000;
    56.62951    44.00000;
    61.21785    45.00000;
    54.38790    46.00000;
    62.93443    47.00000;
    56.65144    48.00000;
    57.13362    49.00000;
    58.29689    50.00000;
    58.91744    51.00000;
    58.50172    52.00000;
    55.22885    53.00000;
    58.30375    54.00000;
    57.43237    55.00000;
    51.69407    56.00000;
    49.93132    57.00000;
    53.70760    58.00000;
    55.39712    59.00000;
    52.89709    60.00000;
    52.31649    61.00000;
    53.98720    62.00000;
    53.54158    63.00000;
    56.45046    64.00000;
    51.32276    65.00000;
    53.11676    66.00000;
    53.28631    67.00000;
    49.80555    68.00000;
    54.69564    69.00000;
    56.41627    70.00000;
    54.59362    71.00000;
    54.38520    72.00000;
    60.15354    73.00000;
    59.78773    74.00000;
    60.49995    75.00000;
    65.43885    76.00000;
    60.70001    77.00000;
    63.71865    78.00000;
    67.77139    79.00000;
    64.70934    80.00000;
    70.78193    81.00000;
    70.38651    82.00000;
    77.22359    83.00000;
    79.52665    84.00000;
    80.13077    85.00000;
    85.67823    86.00000;
    85.20647    87.00000;
    90.24548    88.00000;
    93.61953    89.00000;
    95.86509    90.00000;
    93.46992    91.00000;
    105.8137    92.00000;
    107.8269    93.00000;
    114.0607    94.00000;
    115.5019    95.00000;
    118.5110    96.00000;
    119.6177    97.00000;
    122.1940    98.00000;
    126.9903    99.00000;
    125.7005   100.00000;
    123.7447   101.00000;
    130.6543   102.00000;
    129.7168   103.00000;
    131.8240   104.00000;
    131.8759   105.00000;
    131.9994    106.0000;
    132.1221    107.0000;
    133.4414    108.0000;
    133.8252    109.0000;
    133.6695    110.0000;
    128.2851    111.0000;
    126.5182    112.0000;
    124.7550    113.0000;
    118.4016    114.0000;
    122.0334    115.0000;
    115.2059    116.0000;
    118.7856    117.0000;
    110.7387    118.0000;
    110.2003    119.0000;
    105.17290    120.0000;
    103.44720    121.0000;
    94.54280    122.0000;
    94.40526    123.0000;
    94.57964    124.0000;
    88.76605    125.0000;
    87.28747    126.0000;
    92.50443    127.0000;
    86.27997    128.0000;
    82.44307    129.0000;
    80.47367    130.0000;
    78.36608    131.0000;
    78.74307    132.0000;
    76.12786    133.0000;
    79.13108    134.0000;
    76.76062    135.0000;
    77.60769    136.0000;
    77.76633    137.0000;
    81.28220    138.0000;
    79.74307    139.0000;
    81.97964    140.0000;
    80.02952    141.0000;
    85.95232    142.0000;
    85.96838    143.0000;
    79.94789    144.0000;
    87.17023    145.0000;
    90.50992    146.0000;
    93.23373    147.0000;
    89.14803    148.0000;
    93.11492    149.0000;
    90.34337    150.0000;
    93.69421    151.0000;
    95.74256    152.0000;
    91.85105    153.0000;
    96.74503    154.0000;
    87.60996    155.0000;
    90.47012    156.0000;
    88.11690    157.0000;
    85.70673    158.0000;
    85.01361    159.0000;
    78.53040    160.0000;
    81.34148    161.0000;
    75.19295    162.0000;
    72.66115    163.0000;
    69.85504    164.0000;
    66.29476    165.0000;
    63.58502    166.0000;
    58.33847    167.0000;
    57.50766    168.0000;
    52.80498    169.0000;
    50.79319    170.0000;
    47.03490    171.0000;
    46.47090    172.0000;
    43.09016    173.0000;
    34.11531    174.0000;
    39.28235    175.0000;
    32.68386    176.0000;
    30.44056    177.0000;
    31.98932    178.0000;
    23.63330    179.0000;
    23.69643    180.0000;
    20.26812    181.0000;
    19.07074    182.0000;
    17.59544    183.0000;
    16.08785    184.0000;
    18.94267    185.0000;
    18.61354    186.0000;
    17.25800    187.0000;
    16.62285    188.0000;
    13.48367    189.0000;
    15.37647    190.0000;
    13.47208    191.0000;
    15.96188    192.0000;
    12.32547    193.0000;
    16.33880    194.0000;
    10.438330    195.0000;
    9.628715    196.0000;
    13.12268    197.0000;
    8.772417    198.0000;
    11.76143    199.0000;
    12.55020    200.0000;
    11.33108    201.0000;
    11.20493    202.0000;
    7.816916    203.0000;
    6.800675    204.0000;
    14.26581    205.0000;
    10.66285    206.0000;
    8.911574    207.0000;
    11.56733    208.0000;
    11.58207    209.0000;
    11.59071    210.0000;
    9.730134    211.0000;
    11.44237    212.0000;
    11.22912    213.0000;
    10.172130    214.0000;
    12.50905    215.0000;
    6.201493    216.0000;
    9.019605    217.0000;
    10.80607    218.0000;
    13.09625    219.0000;
    3.914271    220.0000;
    9.567886    221.0000;
    8.038448    222.0000;
    10.231040    223.0000;
    9.367410    224.0000;
    7.695971    225.0000;
    6.118575    226.0000;
    8.793207    227.0000;
    7.796692    228.0000;
    12.45065    229.0000;
    10.61601    230.0000;
    6.001003    231.0000;
    6.765098    232.0000;
    8.764653    233.0000;
    4.586418    234.0000;
    8.390783    235.0000;
    7.209202    236.0000;
    10.012090    237.0000;
    7.327461    238.0000;
    6.525136    239.0000;
    2.840065    240.0000;
    10.323710    241.0000;
    4.790035    242.0000;
    8.376431    243.0000;
    6.263980    244.0000;
    2.705892    245.0000;
    8.362109    246.0000;
    8.983507    247.0000;
    3.362469    248.0000;
    1.182678    249.0000;
    4.875312    250.0000;
    ]

    parameters = [
    96.0        98.0;
    0.009       0.0105;
    103.0       103.0;
    106.0       105.0;
    18.0        20.0;
    72.0        73.0;
    151.0       150.0;
    18.0        20.0;
    ]
    f(x, beta) = beta[1] * exp(-beta[2]*x) + beta[3] * exp(-(x - beta[4])^2 / beta[5]^2) + beta[6]*exp(-(x - beta[7])^2 / beta[8]^2)

     solution = [
     9.9018328406E+01 ,
     1.0994945399E-02 ,
     1.0188022528E+02 ,
     1.0703095519E+02 ,
     2.3578584029E+01 ,
     7.2045589471E+01 ,
     1.5327010194E+02 ,
     1.9525972636E+01 
     ]
    return name, data, parameters, f, solution
end


function DanWood()
    name = "DanWood"
    data = [
    2.138E0        1.309E0;
    3.421E0        1.471E0;
    3.597E0        1.490E0;
    4.340E0        1.565E0;
    4.882E0        1.611E0;
    5.660E0        1.680E0;
    ]
    parameters =[
    1           0.7;
    5           4;
    ]
    f(x, beta) = beta[1] * x^beta[2]
    solution =  [7.6886226176E-01,  3.8604055871E+00]
 

    return name, data, parameters, f, solution
end

function Misra1b()
    name = "Misra1b"
    data =[
    10.07E0      77.6E0;
    14.73E0     114.9E0;
    17.94E0     141.1E0;
    23.93E0     190.8E0;
    29.61E0     239.9E0;
    35.18E0     289.0E0;
    40.02E0     332.8E0;
    44.82E0     378.4E0;
    50.76E0     434.8E0;
    55.05E0     477.3E0;
    61.01E0     536.8E0;
    66.40E0     593.1E0;
    75.47E0     689.1E0;
    81.78E0     760.0E0;
    ]

    parameters = [
    500         300   ;
    0.0001      0.0002;
    ]

    f(x, beta) =  beta[1]* (1 - (1 / (1 + beta[2] * x / 2)^2))
    solution = [3.3799746163E+02, 3.9039091287E-04]


    return name, data, parameters, f, solution
end


function MGH09()
    name = "MGH09"
    data = [
    1.957000E-01    4.000000E+00;
    1.947000E-01    2.000000E+00;
    1.735000E-01    1.000000E+00;
    1.600000E-01    5.000000E-01;
    8.440000E-02    2.500000E-01;
    6.270000E-02    1.670000E-01;
    4.560000E-02    1.250000E-01;
    3.420000E-02    1.000000E-01;
    3.230000E-02    8.330000E-02;
    2.350000E-02    7.140000E-02;
    2.460000E-02    6.250000E-02;
    ]

    parameters = [
    25          0.25 ;
    39          0.39 ;
    41.5        0.415;
    39          0.39 ;
    ]
    f(x, beta) =  beta[1]*(x^2 + x*beta[2]) / (x^2 + x * beta[3] + beta[4])
    solution = [ 
    1.9280693458E-01,     
    1.9128232873E-01,     
    1.2305650693E-01,     
    1.3606233068E-01
    ]    
    return name, data, parameters, f, solution
end


function Thurber()
    name = "Thurber"
    data = [
    80.574E0      -3.067E0;
    84.248E0      -2.981E0;
    87.264E0      -2.921E0;
    87.195E0      -2.912E0;
    89.076E0      -2.840E0;
    89.608E0      -2.797E0;
    89.868E0      -2.702E0;
    90.101E0      -2.699E0;
    92.405E0      -2.633E0;
    95.854E0      -2.481E0;
    100.696E0      -2.363E0;
    101.060E0      -2.322E0;
    401.672E0      -1.501E0;
    390.724E0      -1.460E0;
    567.534E0      -1.274E0;
    635.316E0      -1.212E0;
    733.054E0      -1.100E0;
    759.087E0      -1.046E0;
    894.206E0      -0.915E0;
    990.785E0      -0.714E0;
    1090.109E0      -0.566E0;
    1080.914E0      -0.545E0;
    1122.643E0      -0.400E0;
    1178.351E0      -0.309E0;
    1260.531E0      -0.109E0;
    1273.514E0      -0.103E0;
    1288.339E0       0.010E0;
    1327.543E0       0.119E0;
    1353.863E0       0.377E0;
    1414.509E0       0.790E0;
    1425.208E0       0.963E0;
    1421.384E0       1.006E0;
    1442.962E0       1.115E0;
    1464.350E0       1.572E0;
    1468.705E0       1.841E0;
    1447.894E0       2.047E0;
    1457.628E0       2.200E0;
    ]

    parameters = [
    1000        1300     ;
    1000        1500     ;
    400         500      ;
    40          75       ;
    0.7         1        ;
    0.3         0.4      ;
    0.03        0.05     ;
    ]
    f(x, beta) =  (beta[1] + beta[2]*x + beta[3]*x^2 + beta[4]*x^3) / (1 + beta[5]*x + beta[6]*x^2 + beta[7]*x^3) 
    solution = [
    1.2881396800E+03 ,
     1.4910792535E+03,
     5.8323836877E+02,
     7.5416644291E+01,
     9.6629502864E-01,
     3.9797285797E-01,
     4.9727297349E-02
     ]
    return name, data, parameters, f, solution
end




function BoxBOD()
    name = "BoxBOD"
    data = [
    109             1;
    149             2;
    149             3;
    191             5;
    213             7;
    224            10;
    ]

    parameters = [
    1           100;
    1          0.75;
    ]
    f(x, beta) =  beta[1]* (1 - exp(-beta[2]*x))
    solution = [2.1380940889E+02,  5.4723748542E-01]

    return name, data, parameters, f, solution
end


function Rat42()
    name = "Rat42"
    data = [
    8.930E0        9.000E0;
    10.800E0       14.000E0;
    18.590E0       21.000E0;
    22.330E0       28.000E0;
    39.350E0       42.000E0;
    56.110E0       57.000E0;
    61.730E0       63.000E0;
    64.620E0       70.000E0;
    67.080E0       79.000E0;
    ]


    parameters = [
    100         75;
    1          2.5;
    0.1        0.07;
    ]

    f(x, beta) = beta[1] / (1 + exp(beta[2] - beta[3]*x))
    solution = [ 7.2462237576E+01 , 2.6180768402E+00, 6.7359200066E-02]

    return name, data, parameters, f, solution
end

function MGH10()
    name = "MGH10"
    data = [
    3.478000E+04    5.000000E+01;
    2.861000E+04    5.500000E+01;
    2.365000E+04    6.000000E+01;
    1.963000E+04    6.500000E+01;
    1.637000E+04    7.000000E+01;
    1.372000E+04    7.500000E+01;
    1.154000E+04    8.000000E+01;
    9.744000E+03    8.500000E+01;
    8.261000E+03    9.000000E+01;
    7.030000E+03    9.500000E+01;
    6.005000E+03    1.000000E+02;
    5.147000E+03    1.050000E+02;
    4.427000E+03    1.100000E+02;
    3.820000E+03    1.150000E+02;
    3.307000E+03    1.200000E+02;
    2.872000E+03    1.250000E+02;
    ]

    parameters = [
    2         0.02 ;
    400000      4000 ;
    25000       250    ;
    ]

    f(x, beta) =  beta[1] * exp(beta[2] / (x + beta[3]))
    solution = [5.6096364710E-03, 6.1813463463E+03, 3.4522363462E+02    ]
    return name, data, parameters, f, solution
end




function Eckerle4()
    name = "Eckerle4"
    data = [
    0.0001575E0    400.000000E0;
    0.0001699E0    405.000000E0;
    0.0002350E0    410.000000E0;
    0.0003102E0    415.000000E0;
    0.0004917E0    420.000000E0;
    0.0008710E0    425.000000E0;
    0.0017418E0    430.000000E0;
    0.0046400E0    435.000000E0;
    0.0065895E0    436.500000E0;
    0.0097302E0    438.000000E0;
    0.0149002E0    439.500000E0;
    0.0237310E0    441.000000E0;
    0.0401683E0    442.500000E0;
    0.0712559E0    444.000000E0;
    0.1264458E0    445.500000E0;
    0.2073413E0    447.000000E0;
    0.2902366E0    448.500000E0;
    0.3445623E0    450.000000E0;
    0.3698049E0    451.500000E0;
    0.3668534E0    453.000000E0;
    0.3106727E0    454.500000E0;
    0.2078154E0    456.000000E0;
    0.1164354E0    457.500000E0;
    0.0616764E0    459.000000E0;
    0.0337200E0    460.500000E0;
    0.0194023E0    462.000000E0;
    0.0117831E0    463.500000E0;
    0.0074357E0    465.000000E0;
    0.0022732E0    470.000000E0;
    0.0008800E0    475.000000E0;
    0.0004579E0    480.000000E0;
    0.0002345E0    485.000000E0;
    0.0001586E0    490.000000E0;
    0.0001143E0    495.000000E0;
    0.0000710E0    500.000000E0;
    ]


    parameters = [
    1           1.5 ;
    10           5   ;
    500         450   ;
    ]
    f(x, beta) = beta[1] / beta[2] * exp(-(x - beta[3])^2 / (2*beta[2]^2))
    solution = [1.5543827178E+00    ,   4.0888321754E+00    ,4.5154121844E+02]

    return name, data, parameters, f, solution
end





function Rat43()
    name = "Rat43"
    data =[
    16.08E0     1.0E0;
    33.83E0     2.0E0;
    65.80E0     3.0E0;
    97.20E0     4.0E0;
    191.55E0     5.0E0;
    326.20E0     6.0E0;
    386.87E0     7.0E0;
    520.53E0     8.0E0;
    590.03E0     9.0E0;
    651.92E0    10.0E0;
    724.93E0    11.0E0;
    699.56E0    12.0E0;
    689.96E0    13.0E0;
    637.56E0    14.0E0;
    717.41E0    15.0E0;
    ]

    parameters = [
    100         700;
    10           5;
    1           0.75;
    1           1.3 ;
    ]
    f(x, beta) = beta[1] / (1 + exp(beta[2] - beta[3] * x))^(1/beta[4])
    solution = [
    6.9964151270E+02, 
    5.2771253025E+00,
    7.5962938329E-01,
    1.2792483859E+00
  ]

    return name, data, parameters, f, solution
end




function Bennett5()

    name = "Bennet5"
    data = [
    -34.834702E0      7.447168E0;
    -34.393200E0      8.102586E0;
    -34.152901E0      8.452547E0;
    -33.979099E0      8.711278E0;
    -33.845901E0      8.916774E0;
    -33.732899E0      9.087155E0;
    -33.640301E0      9.232590E0;
    -33.559200E0      9.359535E0;
    -33.486801E0      9.472166E0;
    -33.423100E0      9.573384E0;
    -33.365101E0      9.665293E0;
    -33.313000E0      9.749461E0;
    -33.260899E0      9.827092E0;
    -33.217400E0      9.899128E0;
    -33.176899E0      9.966321E0;
    -33.139198E0     10.029280E0;
    -33.101601E0     10.088510E0;
    -33.066799E0     10.144430E0;
    -33.035000E0     10.197380E0;
    -33.003101E0     10.247670E0;
    -32.971298E0     10.295560E0;
    -32.942299E0     10.341250E0;
    -32.916302E0     10.384950E0;
    -32.890202E0     10.426820E0;
    -32.864101E0     10.467000E0;
    -32.841000E0     10.505640E0;
    -32.817799E0     10.542830E0;
    -32.797501E0     10.578690E0;
    -32.774300E0     10.613310E0;
    -32.757000E0     10.646780E0;
    -32.733799E0     10.679150E0;
    -32.716400E0     10.710520E0;
    -32.699100E0     10.740920E0;
    -32.678799E0     10.770440E0;
    -32.661400E0     10.799100E0;
    -32.644001E0     10.826970E0;
    -32.626701E0     10.854080E0;
    -32.612202E0     10.880470E0;
    -32.597698E0     10.906190E0;
    -32.583199E0     10.931260E0;
    -32.568699E0     10.955720E0;
    -32.554298E0     10.979590E0;
    -32.539799E0     11.002910E0;
    -32.525299E0     11.025700E0;
    -32.510799E0     11.047980E0;
    -32.499199E0     11.069770E0;
    -32.487598E0     11.091100E0;
    -32.473202E0     11.111980E0;
    -32.461601E0     11.132440E0;
    -32.435501E0     11.152480E0;
    -32.435501E0     11.172130E0;
    -32.426800E0     11.191410E0;
    -32.412300E0     11.210310E0;
    -32.400799E0     11.228870E0;
    -32.392101E0     11.247090E0;
    -32.380501E0     11.264980E0;
    -32.366001E0     11.282560E0;
    -32.357300E0     11.299840E0;
    -32.348598E0     11.316820E0;
    -32.339901E0     11.333520E0;
    -32.328400E0     11.349940E0;
    -32.319698E0     11.366100E0;
    -32.311001E0     11.382000E0;
    -32.299400E0     11.397660E0;
    -32.290699E0     11.413070E0;
    -32.282001E0     11.428240E0;
    -32.273300E0     11.443200E0;
    -32.264599E0     11.457930E0;
    -32.256001E0     11.472440E0;
    -32.247299E0     11.486750E0;
    -32.238602E0     11.500860E0;
    -32.229900E0     11.514770E0;
    -32.224098E0     11.528490E0;
    -32.215401E0     11.542020E0;
    -32.203800E0     11.555380E0;
    -32.198002E0     11.568550E0;
    -32.189400E0     11.581560E0;
    -32.183601E0     11.594420E0;
    -32.174900E0     11.607121E0;
    -32.169102E0     11.619640E0;
    -32.163300E0     11.632000E0;
    -32.154598E0     11.644210E0;
    -32.145901E0     11.656280E0;
    -32.140099E0     11.668200E0;
    -32.131401E0     11.679980E0;
    -32.125599E0     11.691620E0;
    -32.119801E0     11.703130E0;
    -32.111198E0     11.714510E0;
    -32.105400E0     11.725760E0;
    -32.096699E0     11.736880E0;
    -32.090900E0     11.747890E0;
    -32.088001E0     11.758780E0;
    -32.079300E0     11.769550E0;
    -32.073502E0     11.780200E0;
    -32.067699E0     11.790730E0;
    -32.061901E0     11.801160E0;
    -32.056099E0     11.811480E0;
    -32.050301E0     11.821700E0;
    -32.044498E0     11.831810E0;
    -32.038799E0     11.841820E0;
    -32.033001E0     11.851730E0;
    -32.027199E0     11.861550E0;
    -32.024300E0     11.871270E0;
    -32.018501E0     11.880890E0;
    -32.012699E0     11.890420E0;
    -32.004002E0     11.899870E0;
    -32.001099E0     11.909220E0;
    -31.995300E0     11.918490E0;
    -31.989500E0     11.927680E0;
    -31.983700E0     11.936780E0;
    -31.977900E0     11.945790E0;
    -31.972099E0     11.954730E0;
    -31.969299E0     11.963590E0;
    -31.963501E0     11.972370E0;
    -31.957701E0     11.981070E0;
    -31.951900E0     11.989700E0;
    -31.946100E0     11.998260E0;
    -31.940300E0     12.006740E0;
    -31.937401E0     12.015150E0;
    -31.931601E0     12.023490E0;
    -31.925800E0     12.031760E0;
    -31.922899E0     12.039970E0;
    -31.917101E0     12.048100E0;
    -31.911301E0     12.056170E0;
    -31.908400E0     12.064180E0;
    -31.902599E0     12.072120E0;
    -31.896900E0     12.080010E0;
    -31.893999E0     12.087820E0;
    -31.888201E0     12.095580E0;
    -31.885300E0     12.103280E0;
    -31.882401E0     12.110920E0;
    -31.876600E0     12.118500E0;
    -31.873699E0     12.126030E0;
    -31.867901E0     12.133500E0;
    -31.862101E0     12.140910E0;
    -31.859200E0     12.148270E0;
    -31.856300E0     12.155570E0;
    -31.850500E0     12.162830E0;
    -31.844700E0     12.170030E0;
    -31.841801E0     12.177170E0;
    -31.838900E0     12.184270E0;
    -31.833099E0     12.191320E0;
    -31.830200E0     12.198320E0;
    -31.827299E0     12.205270E0;
    -31.821600E0     12.212170E0;
    -31.818701E0     12.219030E0;
    -31.812901E0     12.225840E0;
    -31.809999E0     12.232600E0;
    -31.807100E0     12.239320E0;
    -31.801300E0     12.245990E0;
    -31.798401E0     12.252620E0;
    -31.795500E0     12.259200E0;
    -31.789700E0     12.265750E0;
    -31.786800E0     12.272240E0;
    ]

    parameters = [
    -2000       -1500;
    50          45;
    0.8         0.85;
    ]
    f(x, beta) = beta[1] *(beta[2] + x)^(-1/beta[3])
    solution = [    -2.5235058043E+03,4.6736564644E+01,9.3218483193E-01]
    return name, data, parameters, f, solution
end


function ff!(fcur, x, f, data)
    for i in 1:size(data, 1)
        fcur[i] = data[i, 1] - f(data[i, 2], x)
    end
end


tests =  [misra1a, Chwirut2, Chwitrut1, Lanczos3, Gauss1, Gauss2, DanWood, Misra1b, MGH09, Thurber, BoxBOD, Rat42, MGH10, Eckerle4, Rat43, Bennett5]

for (optimizer, optimizer_abbr) in ((LeastSquaresOptim.Dogleg, :dl), (LeastSquaresOptim.LevenbergMarquardt, :lm))
    n = 0
    N = 0
    for test in tests
        local name, data, parameters, f, solution = test()
      #  @show solution
        for j in 1:size(parameters, 2)
            local nls = LeastSquaresProblem(x = parameters[:, j], f! = (fcur, x) -> ff!(fcur, x, f, data), output_length = size(data, 1))
            local r = optimize!(nls, optimizer(LeastSquaresOptim.QR()), x_tol = 1e-50, f_tol = 1e-36, g_tol = 1e-50)
            n += norm(r.minimizer - solution) <= 1e-3
            N += 1
            @test !isnan(mean(r.minimizer) )
        end
    end
    println("strd $optimizer_abbr $n / $N")
end

