# oneliner fast build PLOTDOCS_SUFFIX='' PLOTDOCS_PACKAGES='UnicodePlots' PLOTDOCS_EXAMPLES='1' julia --project make.jl
import Pkg; Pkg.precompile()

using Plots, RecipesBase, RecipesPipeline
using DemoCards, Literate, Documenter

import OrderedCollections
import GraphRecipes
import PythonPlot
import StableRNGs
import StatsPlots
import MacroTools
import DataFrames
import PlotThemes
import Dates
import JSON
import Glob

PythonPlot.pygui(false)  # prevent segfault on event loop in ci

suffix = get(ENV, "PLOTDOCS_SUFFIX", "")
const WRK_DIR = joinpath(@__DIR__, "work" * suffix)
const BLD_DIR = joinpath(@__DIR__, "build" * suffix)
const SRC_DIR = joinpath(@__DIR__, "src")
const GEN_DIR = joinpath(WRK_DIR, "generated")
const BRANCH = ("master", "v2")[2]  # transition to v2
const ASSETS = "assets"

const ATTRIBUTE_SEARCH = Dict{String, Any}()  # search terms

# monkey patch `Documenter` - note that this could break on minor `Documenter` releases
@eval Documenter.Writers.HTMLWriter domify(dctx::DCtx) = begin
    ctx, navnode = dctx.ctx, dctx.navnode
    return map(getpage(ctx, navnode).mdast.children) do node
        rec = SearchRecord(ctx, navnode, node, node.element)
        ############################################################
        # begin addition
        info = "[src=$(rec.src) fragment=$(rec.fragment) title=$(rec.title) page_title=$(rec.page_title)]"
        if (m = match(r"generated/attributes_(\w+)", lowercase(rec.src))) ≢ nothing
            # fix attributes search terms: `Series`, `Plot`, `Subplot` and `Axis` (github.com/JuliaPlots/Plots.jl/issues/2337)
            @info "$info: fix attribute search" maxlog = 10
            for (attr, alias) in $(ATTRIBUTE_SEARCH)[first(m.captures)]
                push!(
                    ctx.search_index,
                    SearchRecord(rec.src, rec.page, rec.fragment, rec.category, rec.title, rec.page_title, attr * ' ' * alias)
                )
            end
        else
            add_to_index = if (m = match(r"gallery/(\w+)/", lowercase(rec.src))) ≢ nothing
                first(m.captures) == "gr"  # only add `GR` gallery pages to `search_index` (github.com/JuliaPlots/Plots.jl/issues/4157)
            else
                true
            end
            if add_to_index
                push!(ctx.search_index, rec)
            else
                @info "$info: skip adding to `search_index`" maxlog = 10
            end
        end
        # end addition
        ############################################################
        domify(dctx, node, node.element)
    end
end

@eval DemoCards get_logopath() = $(joinpath(SRC_DIR, ASSETS, "axis_logo_600x400.png"))

# ----------------------------------------------------------------------

edit_url(args...) = "https://github.com/JuliaPlots/Plots.jl/blob/$BRANCH/docs/" * if length(args) == 0
    "make.jl"
else
    joinpath(basename(WRK_DIR), args...)
end

autogenerated() = "(Automatically generated: " * Dates.format(Dates.now(), Dates.RFC1123Format) * ')'

author() = "[Plots.jl](https://github.com/JuliaPlots/Plots.jl)"

recursive_rmlines(x) = x
function recursive_rmlines(x::Expr)
    x = MacroTools.rmlines(x)
    x.args .= recursive_rmlines.(x.args)
    return x
end

pretty_print_expr(io::IO, expr::Expr) = if expr.head ≡ :block
    foreach(arg -> println(io, arg), recursive_rmlines(expr).args)
else
    println(io, recursive_rmlines(expr))
end

markdown_code_to_string(arr, prefix = "") =
    surround_backticks(prefix, join(sort(map(string, arr)), "`, `$prefix"))

markdown_symbols_to_string(arr) = isempty(arr) ? "" : markdown_code_to_string(arr, ":")

# ----------------------------------------------------------------------

function generate_cards(
        prefix::AbstractString, backend::Symbol, slice;
        skip = get(Plots._backend_skips, backend, Int[]),
        debug = false
    )
    @show backend
    # create folder: for each backend we generate a DemoSection "generated" under "gallery"
    cards_path = let dn = joinpath(prefix, string(backend), "generated" * suffix)
        isdir(dn) && rm(dn; recursive = true)
        mkpath(dn)
    end
    sec_config = Dict{String, Any}("order" => [])
    needs_rng_fix = Dict{Int, Bool}()

    for (i, example) in enumerate(Plots._examples)
        i ∈ skip && continue
        i ∈ slice || continue
        # write out the header, description, code block, and image link
        jl_name = "$backend-$(Plots.ref_name(i)).jl"
        jl = PipeBuffer()

        # DemoCards YAML frontmatter
        # https://johnnychen94.github.io/DemoCards.jl/stable/quickstart/usage_example/julia_demos/1.julia_demo/#juliademocard_example
        svg_ready_backends = if false
            (:gr, :pythonplot, :pgfplotsx, :plotlyjs, :gaston)  # NOTE: could increase docs repo size ...
        else
            ()
        end
        cover_name = "$(backend)_$(Plots.ref_name(i))"
        cover_path = let cover_file = cover_name * if i ∈ Plots._animation_examples
                ".gif"
            elseif backend ∈ svg_ready_backends
                ".svg"
            else
                ".png"
            end
            joinpath(ASSETS, cover_file)
        end
        if !isempty(example.header)
            push!(sec_config["order"], jl_name)
            # start a new demo file
            debug && @info "generate demo $(example.header |> repr) - writing `$jl_name`"

            extra = if backend ≡ :unicodeplots
                "import FileIO, FreeType  #hide"  # weak deps for png export
            else
                ""
            end
            write(
                jl, """
                # ---
                # title: $(example.header)
                # id: $cover_name
                # cover: $cover_path
                # author: "$(author())"
                # description: ""
                # date: $(Dates.now())
                # execute: true
                # ---
                using Plots
                $backend()
                $extra
                Plots.reset_defaults()  #hide
                using StableRNGs  #hide
                rng = StableRNG($(Plots.PLOTS_SEED))  #hide
                nothing  #hide
                """
            )
        end
        # DemoCards use Literate.jl syntax with extra leading `#` as markdown lines
        write(jl, "# $(replace(example.desc, "\n" => "\n  # "))\n")
        isnothing(example.imports) || pretty_print_expr(jl, example.imports)
        needs_rng_fix[i] = (exprs_rng = Plots.replace_rand(example.exprs)) != example.exprs
        pretty_print_expr(jl, exprs_rng)

        # NOTE: the supported `Literate.jl` syntax is `#src` and `#hide` NOT `# src` !!
        # from the docs: """
        # #src and #hide are quite similar. The only difference is that #src lines are filtered out before execution (if execute=true) and #hide lines are filtered out after execution.
        # """
        # this command creates the card cover file, NOT the output of the `@example` block in the `index.html` file !
        cover_cmd = if i ∈ Plots._animation_examples
            "Plots.gif(anim, $(cover_path |> repr))"
        elseif backend ∈ svg_ready_backends
            "Plots.svg($(cover_path |> repr))"
        else
            "Plots.png($(cover_path |> repr))"
        end
        # this command creates the output of the `@example` block in the `index.html` card file
        show_cmd = if i ∈ Plots._animation_examples
            "Plots.gif(anim)"
        elseif backend ≡ :plotlyjs
            # FIXME: failing to render the html script outputted by :plotlyjs so instead include the cover .svg file
            """
            nothing  #hide
            # ![$cover_name]($cover_path)
            """
        else
            "current()"  # triggers MIME("text/html"), see Plots._best_html_output_type (mostly `:svg` and `:html`)
        end
        write(
            jl, """
            mkpath($(ASSETS |> repr))  #src
            $cover_cmd  #src
            $show_cmd  #hide
            """
        )
        card_jl = read(jl, String)
        debug && @info card_jl

        fn, mode = if isempty(example.header)
            "$backend-$(Plots.ref_name(i - 1)).jl", "a"  # continued example
        else
            jl_name, "w"
        end
        card = joinpath(cards_path, fn)
        # @info "writing" card
        open(io -> write(io, card_jl), card, mode)
        # DEBUG: sometimes the generated file is still empty when passing to `DemoCards.makedemos`
        sleep(0.01)
    end
    # insert attributes page
    # TODO(johnnychen): make this part of the page template
    attr_name = string(backend, ".jl")
    open(joinpath(cards_path, attr_name), "w") do jl
        pkg = Plots._backend_instance(backend)
        write(
            jl, """
            # ---
            # title: Supported attribute values
            # id: $(backend)_attributes
            # hidden: true
            # author: "$(author())"
            # date: $(Dates.now())
            # ---

            # - Supported arguments: $(markdown_code_to_string(collect(Plots.supported_attrs(pkg))))
            # - Supported values for linetype: $(markdown_symbols_to_string(Plots.supported_seriestypes(pkg)))
            # - Supported values for linestyle: $(markdown_symbols_to_string(Plots.supported_styles(pkg)))
            # - Supported values for marker: $(markdown_symbols_to_string(Plots.supported_markers(pkg)))
            """
        )
    end
    open(joinpath(cards_path, "config.json"), "w") do config
        sec_config["title"] = ""  # avoid `# Generated` section in gallery
        sec_config["description"] = "[Supported attributes](@ref $(backend)_attributes)"
        push!(sec_config["order"], attr_name)
        write(config, JSON.json(sec_config))
    end
    return needs_rng_fix
end

# tables detailing the features that each backend supports
function make_support_df(allvals, func; default_backends)
    vals = sort(collect(allvals)) # rows
    bs = sort(collect(default_backends))
    df = DataFrames.DataFrame(keys = vals)

    for be in bs # cols
        be_supported_vals = fill("", length(vals))
        for (i, val) in enumerate(vals)
            be_supported_vals[i] = if func == Plots.supported_seriestypes
                stype = Plots.seriestype_supported(Plots._backend_instance(be), val)
                stype ≡ :native ? "✅" : (stype ≡ :no ? "" : "🔼")
            else
                val ∈ func(Plots._backend_instance(be)) ? "✅" : ""
            end
        end
        df[!, be] = be_supported_vals
    end
    return df
end

function generate_supported_markdown(; default_backends)
    supported_args = OrderedCollections.OrderedDict(
        "Keyword Arguments" => (Plots._all_args, Plots.supported_attrs),
        "Markers" => (Plots._allMarkers, Plots.supported_markers),
        "Line Styles" => (Plots._allStyles, Plots.supported_styles),
        "Scales" => (Plots._allScales, Plots.supported_scales)
    )
    return open(joinpath(GEN_DIR, "supported.md"), "w") do md
        write(
            md, """
            ```@meta
            EditURL = "$(edit_url())"
            ```

            ## [Series Types](@id supported)

            Key:

            - ✅ the series type is natively supported by the backend.
            - 🔼 the series type is supported through series recipes.

            ```@raw html
            $(to_html(make_support_df(Plots.all_seriestypes(), Plots.supported_seriestypes; default_backends)))
            ```
            """
        )
        for (header, args) in supported_args
            write(
                md, """

                ## $header

                ```@raw html
                $(to_html(make_support_df(args...; default_backends)))
                ```
                """
            )
        end
        write(md, '\n' * autogenerated())
    end
end

function make_attr_df(ktype::Symbol, defs::KW)
    n = length(defs)
    df = DataFrames.DataFrame(
        Attribute = fill("", n),
        Aliases = fill("", n),
        Default = fill("", n),
        Type = fill("", n),
        Description = fill("", n),
    )
    for (i, (k, def)) in enumerate(defs)
        type, desc = get(Plots._arg_desc, k, (Any, ""))

        aliases = sort(collect(keys(filter(p -> p.second == k, Plots._keyAliases))))
        df.Attribute[i] = string(k)
        df.Aliases[i] = join(aliases, ", ")
        df.Default[i] = show_default(def)
        df.Type[i] = string(type)
        df.Description[i] = string(desc)
    end
    sort!(df, [:Attribute])
    return df
end

surround_backticks(args...) = '`' * string(args...) * '`'
show_default(x) = surround_backticks(x)
show_default(x::Symbol) = surround_backticks(":$x")

function generate_attr_markdown(c)
    attribute_texts = Dict(
        :Series => "These attributes apply to individual series (lines, scatters, heatmaps, etc)",
        :Plot => "These attributes apply to the full Plot. (A Plot contains a tree-like layout of Subplots)",
        :Subplot => "These attributes apply to settings for individual Subplots.",
        :Axis => """
            These attributes apply by default to all Axes in a Subplot (for example the `subplot[:xaxis]`).
            !!! info
                You can also specific the x, y, or z axis for each of these attributes by prefixing the attribute name with x, y, or z
                (for example `xmirror` only sets the mirror attribute for the x axis).
            """,
    )
    attribute_defaults = Dict(
        :Series => Plots._series_defaults,
        :Plot => Plots._plot_defaults,
        :Subplot => Plots._subplot_defaults,
        :Axis => Plots._axis_defaults,
    )

    df = make_attr_df(c, attribute_defaults[c])
    cstr = lowercase(string(c))
    ATTRIBUTE_SEARCH[cstr] = collect(zip(df.Attribute, df.Aliases))

    return open(joinpath(GEN_DIR, "attributes_$cstr.md"), "w") do md
        write(
            md, """
            ```@meta
            EditURL = "$(edit_url())"
            ```
            ### $c

            $(attribute_texts[c])

            ```@raw html
            $(to_html(df))
            ```

            $(autogenerated())
            """
        )
    end
end

generate_attr_markdown() =
    foreach(c -> generate_attr_markdown(c), (:Series, :Plot, :Subplot, :Axis))

function generate_graph_attr_markdown()
    df = DataFrames.DataFrame(
        Attribute = [
            "dim",
            "T",
            "curves",
            "curvature_scalar",
            "root",
            "node_weights",
            "names",
            "fontsize",
            "nodeshape",
            "nodesize",
            "nodecolor",
            "x, y, z",
            "method",
            "func",
            "shorten",
            "axis_buffer",
            "layout_kw",
            "edgewidth",
            "edgelabel",
            "edgelabel_offset",
            "self_edge_size",
            "edge_label_box",
        ],
        Aliases = [
            "",
            "",
            "",
            "curvaturescalar, curvature",
            "",
            "nodeweights",
            "",
            "",
            "node_shape",
            "node_size",
            "marker_color",
            "x",
            "",
            "",
            "shorten_edge",
            "axisbuffer",
            "",
            "edge_width, ew",
            "edge_label, el",
            "edgelabeloffset, elo",
            "selfedgesize, ses",
            "edgelabelbox, edgelabel_box, elb",
        ],
        Default = [
            "2",
            "Float64",
            "true",
            "0.05",
            ":top",
            "nothing",
            "[]",
            "7",
            ":hexagon",
            "0.1",
            "1",
            "nothing",
            ":stress",
            "get(_graph_funcs, method, by_axis_local_stress_graph)",
            "0.0",
            "0.2",
            "Dict{Symbol,Any}()",
            "(s, d, w) -> 1",
            "nothing",
            "0.0",
            "0.1",
            "true",
        ],
        Description = [
            "The number of dimensions in the visualization.",
            "The data type for the coordinates of the graph nodes.",
            "Whether or not edges are curved. If `curves == true`, then the edge going from node \$s\$ to node \$d\$ will be defined by a cubic spline passing through three points: (i) node \$s\$, (ii) a point `p` that is distance `curvature_scalar` from the average of node \$s\$ and node \$d\$ and (iii) node \$d\$.",
            "A scalar that defines how much edges curve, see `curves` for more explanation.",
            "For displaying trees, choose from `:top`, `:bottom`, `:left`, `:right`. If you choose `:top`, then the tree will be plotted from the top down.",
            "The weight of the nodes given by a list of numbers. If `node_weights != nothing`, then the size of the nodes will be scaled by the `node_weights` vector.",
            "Names of the nodes given by a list of objects that can be parsed into strings. If the list is smaller than the number of nodes, then GraphRecipes will cycle around the list.",
            "Font size for the node labels and the edge labels.",
            "Shape of the nodes, choose from `:hexagon`, `:circle`, `:ellipse`, `:rect` or `:rectangle`.",
            "The size of nodes in the plot coordinates. Note that if `names` is not empty, then nodes will be scaled to fit the labels inside them.",
            "The color of the nodes. If `nodecolor` is an integer, then it will be taken from the current color palette. Otherwise, the user can pass any color that would be recognised by the Plots `color` attribute.",
            "The coordinates of the nodes.",
            "The method that GraphRecipes uses to produce an optimal layout, choose from `:spectral`, `:sfdp`, `:circular`, `:shell`, `:stress`, `:spring`, `:tree`, `:buchheim`, `:arcdiagram` or `:chorddiagram`. See [NetworkLayout](https://github.com/JuliaGraphs/NetworkLayout.jl) for further details.",
            "A layout algorithm that can be passed in by the user.",
            "An amount to shorten edges by.",
            "Increase the `xlims` and `ylims`/`zlims` of the plot. Can be useful if part of the graph sits outside of the default view.",
            "A list of keywords to be passed to the layout algorithm, see [NetworkLayout](https://github.com/JuliaGraphs/NetworkLayout.jl) for a list of keyword arguments for each algorithm.",
            "The width of the edge going from \$s\$ to node \$d\$ with weight \$w\$.",
            "A dictionary of `(s, d) => label`, where `s` is an integer for the source node, `d` is an integer for the destiny node and `label` is the desired label for the given edge. Alternatively the user can pass a vector or a matrix describing the edge labels. If you use a vector or matrix, then either `missing`, `false`, `nothing`, `NaN` or `\"\"` values will not be displayed. In the case of multigraphs, triples can be used to define edges.",
            "The distance between edge labels and edges.",
            "The size of self edges.",
            "A box around edge labels that avoids intersections between edge labels and the edges that they are labeling.",
        ]
    )
    return open(joinpath(GEN_DIR, "graph_attributes.md"), "w") do md
        write(
            md, """
            ```@meta
            EditURL = "$(edit_url())"
            ```
            # [Graph Attributes](@id graph_attributes)

            Where possible, GraphRecipes will adopt attributes from Plots.jl to format visualizations.
            For example, the `linewidth` attribute from Plots.jl has the same effect in `GraphRecipes`.
            In order to give the user control over the layout of the graph visualization,
            `GraphRecipes` provides a number of keyword arguments (attributes).
            Here we describe those attributes alongside their default values.

            ```@raw html
            $(to_html(df))
            ```
            \n
            ## Aliases
            Certain keyword arguments have aliases, so `GraphRecipes` does "what you mean, not what you say".

            So for example, `nodeshape=:rect` and `node_shape=:rect` are equivalent.
            To see the available aliases, type `GraphRecipes.graph_aliases`.
            If you are unhappy with the provided aliases, then you can add your own:
            ```julia
            using GraphRecipes, Plots

            push!(GraphRecipes.graph_aliases[:nodecolor],:nc)

            # These two calls produce the same plot, modulo some randomness in the layout.
            plot(graphplot([0 1; 0 0]; nodecolor=:red), graphplot([0 1; 0 0]; nc=:red))
            ```

            $(autogenerated())
            """
        )
    end
end

generate_colorschemes_markdown() = open(joinpath(GEN_DIR, "colorschemes.md"), "w") do md
    write(
        md, """
        ```@meta
        EditURL = "$(edit_url())"
        ```
        """
    )
    foreach(line -> write(md, line * '\n'), readlines(joinpath(SRC_DIR, "colorschemes.md")))
    write(
        md, """
        ## misc

        These colorschemes are not defined or provide different colors in ColorSchemes.jl
        They are kept for compatibility with Plots behavior before v1.1.0.
        """
    )
    write(md, "```@raw html\n")
    ks = [:default; sort(collect(keys(PlotUtils.MISC_COLORSCHEMES)))]
    write(md, to_html(make_colorschemes_df(ks); allow_html_in_cells = true))
    write(md, "\n```\n\nThe following colorschemes are defined by ColorSchemes.jl.\n\n")
    for cs in ("cmocean", "scientific", "matplotlib", "colorbrewer", "gnuplot", "colorcet", "seaborn", "general")
        ks = sort([k for (k, v) in PlotUtils.ColorSchemes.colorschemes if occursin(cs, v.category)])
        write(md, "\n## $cs\n\n```@raw html\n")
        write(md, to_html(make_colorschemes_df(ks); allow_html_in_cells = true))
        write(md, "\n```\n")
    end
end

function colors_svg(cs, w, h)
    n = length(cs)
    ws = min(w / n, h)
    # NOTE: html tester, codebeautify.org/htmlviewer or htmledit.squarefree.com
    html = replace(
        """
        <?xml version="1.0" encoding="UTF-8"?>
        <!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN"
         "https://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
        <svg xmlns="https://www.w3.org/2000/svg" version="1.1"
             width="$(n * ws)mm" height="$(h)mm"
             viewBox="0 0 $n 1" preserveAspectRatio="none"
             shape-rendering="crispEdges" stroke="none">
        """, "\n" => " "
    )  # NOTE: no linebreaks (because those break html code)
    for (i, c) in enumerate(cs)
        html *= """<rect width="$(ws)mm" height="$(h)mm" x="$(i - 1)" y="0" fill="#$(hex(convert(RGB, c)))" />"""
    end
    return html *= "</svg>"
end

function make_colorschemes_df(ks)
    n = length(ks)
    df = DataFrames.DataFrame(
        Name = fill("", n),
        Palette = fill("", n),
        Gradient = fill("", n),
    )
    len, w, h = 100, 60, 5
    for (i, k) in enumerate(ks)
        p = palette(k)
        cg = cgrad(k)[range(0, 1, length = len)]
        cp = length(p) ≤ len ? color_list(p) : cg
        df.Name[i] = string(':', k)
        df.Palette[i] = colors_svg(cp, w, h)
        df.Gradient[i] = colors_svg(cg, w, h)
    end
    return df
end

# ----------------------------------------------------------------------

function to_html(df::DataFrames.AbstractDataFrame; table = ["font-size" => "12px"], kw...)
    io = PipeBuffer()  # NOTE: `DataFrames` exports `PrettyTables`
    style = DataFrames.PrettyTables.HtmlTableStyle(; table)
    show(
        IOContext(io, :limit => false, :compact => false), MIME"text/html"(), df;
        show_row_number = false, summary = false, eltypes = false, style,
        kw...
    )
    return read(io, String)
end

function main(args)
    args = @. Symbol(args)
    default_build_cmds = [:generate, :gallery, :make, :deploy]
    build_cmds = length(args) > 0 ? args : default_build_cmds
    :all ∈ build_cmds && (build_cmds = default_build_cmds)
    if :none ∈ build_cmds
        Pkg.precompile()
        return
    end
    @show build_cmds

    get!(ENV, "MPLBACKEND", "agg")  # set matplotlib gui backend
    get!(ENV, "GKSwstype", "nul")  # disable default GR ws

    # cleanup
    isdir(WRK_DIR) && rm(WRK_DIR; recursive = true)
    isdir(BLD_DIR) && rm(BLD_DIR; recursive = true)
    mkpath(GEN_DIR)

    # initialize all backends
    gr()
    pythonplot()
    plotlyjs()
    pgfplotsx()
    unicodeplots()
    gaston()
    inspectdr()

    # NOTE: for a faster representative test build use `PLOTDOCS_PACKAGES='GR' PLOTDOCS_EXAMPLES='1'`
    all_packages = "GR PythonPlot PlotlyJS PGFPlotsX UnicodePlots Gaston InspectDR"
    packages = get(ENV, "PLOTDOCS_PACKAGES", "ALL")
    packages = let val = packages == "ALL" ? all_packages : packages
        Symbol.(filter(!isempty, strip.(split(val))))
    end
    packages_backends = NamedTuple(p => Symbol(lowercase(string(p))) for p in packages)
    backends = values(packages_backends) |> collect

    @info "selected packages: $packages"
    @info "selected backends: $backends"

    slice = parse.(Int, split(get(ENV, "PLOTDOCS_EXAMPLES", "")))
    slice = (len_sl = length(slice)) == 0 ? range(1; stop = length(Plots._examples)) : slice
    @info "selected examples: $slice"

    debug = length(packages) ≤ 1 || 1 < len_sl ≤ 3

    work_dir = basename(WRK_DIR)
    bld_dir = basename(BLD_DIR)
    src_dir = basename(SRC_DIR)
    @show debug SRC_DIR WRK_DIR BLD_DIR

    if :generate ∈ build_cmds
        @info "generate markdown"
        @time "generate markdown" begin
            generate_attr_markdown()
            generate_supported_markdown(; default_backends = backends)
            generate_graph_attr_markdown()
            generate_colorschemes_markdown()
        end
    end

    for (pkg, dest) in (
            (PlotThemes, "plotthemes.md"),
            (StatsPlots, "statsplots.md"),
        )
        cp(pkgdir(pkg, "README.md"), joinpath(GEN_DIR, dest); force = true)
    end

    gallery = Pair{String, String}[]
    gallery_assets, gallery_callbacks, user_gallery = map(_ -> [], 1:3)
    needs_rng_fix = Dict{Symbol, Any}()
    if :gallery ∈ build_cmds
        @info "gallery"

        @time "gallery" for pkg in packages
            be = packages_backends[pkg]
            needs_rng_fix[pkg] = generate_cards(joinpath(@__DIR__, "gallery"), be, slice; debug)
            let (path, cb, asset) = makedemos(
                    joinpath(@__DIR__, "gallery", string(be));
                    root = @__DIR__, src = joinpath(work_dir, "gallery"), edit_branch = BRANCH
                )
                push!(gallery, string(pkg) => joinpath("gallery", path))
                push!(gallery_callbacks, cb)
                push!(gallery_assets, asset)
            end
        end
        if !debug
            user_gallery, cb, assets = makedemos(
                joinpath("user_gallery");
                root = @__DIR__, src = work_dir, edit_branch = BRANCH
            )
            push!(gallery_callbacks, cb)
            push!(gallery_assets, assets)
            unique!(gallery_assets)
            @show user_gallery gallery_assets
        end
    end

    pages = Any["Home" => "index.md"]
    if debug
        :gallery ∈ build_cmds && push!(pages, "Gallery" => gallery)
        :generate ∈ build_cmds && push!(pages, "Series Attributes" => "generated/attributes_series.md")
    else  # release
        push!(
            pages,
            "Getting Started" => [
                "Installation" => "install.md",
                "Basics" => "basics.md",
                "Tutorial" => "tutorial.md",
                "Series Types" => [
                    "Contour Plots" => "series_types/contour.md",
                    "Histograms" => "series_types/histogram.md",
                ],
            ]
        )
        :generate ∈ build_cmds && push!(
            pages,
            "Manual" => [
                "Input Data" => "input_data.md",
                "Output" => "output.md",
                "Attributes" => "attributes.md",
                "Series Attributes" => "generated/attributes_series.md",
                "Plot Attributes" => "generated/attributes_plot.md",
                "Subplot Attributes" => "generated/attributes_subplot.md",
                "Axis Attributes" => "generated/attributes_axis.md",
                "Layouts" => "layouts.md",
                "Recipes" => [
                    "Overview" => "recipes.md",
                    "RecipesBase" => [
                        "Home" => "RecipesBase/index.md",
                        "Recipes Syntax" => "RecipesBase/syntax.md",
                        "Recipes Types" => "RecipesBase/types.md",
                        "Internals" => "RecipesBase/internals.md",
                        "Public API" => "RecipesBase/api.md",
                    ],
                    "RecipesPipeline" => [
                        "Home" => "RecipesPipeline/index.md",
                        "Public API" => "RecipesPipeline/api.md",
                    ],
                ],
                "Colors" => "colors.md",
                "ColorSchemes" => "generated/colorschemes.md",
                "Animations" => "animations.md",
                "Themes" => "generated/plotthemes.md",
                "Backends" => "backends.md",
                "Supported Attributes" => "generated/supported.md",
            ],
        )
        push!(
            pages,
            "Learning" => "learning.md",
            "Contributing" => "contributing.md"
        )
        :generate ∈ build_cmds && push!(
            pages,
            "Ecosystem" => [
                "StatsPlots" => "generated/statsplots.md",
                "GraphRecipes" => [
                    "Introduction" => "GraphRecipes/introduction.md",
                    "Examples" => "GraphRecipes/examples.md",
                    "Attributes" => "generated/graph_attributes.md",
                ],
                "UnitfulExt" => [
                    "Introduction" => "UnitfulExt/unitfulext.md",
                    "Examples" => [
                        "Simple" => "generated/unitfulext_examples.md",
                        "Plots" => "generated/unitfulext_plots.md",
                    ],
                ],
                "Overview" => "ecosystem.md",
            ]
        )
        push!(pages, "Advanced Topics" => ["Plotting pipeline" => "pipeline.md"])
        :generate ∈ build_cmds && push!(
            pages,
            "Gallery" => gallery,
            "User Gallery" => user_gallery
        )
        push!(pages, "API" => "api.md")
    end

    # those will be built pages - to skip some pages, comment them above
    selected_pages = []
    collect_pages!(p::Pair) = if p.second isa AbstractVector
        collect_pages!(p.second)
    else
        push!(selected_pages, basename(p.second))
    end
    collect_pages!(v::AbstractVector) = foreach(collect_pages!, v)

    collect_pages!(pages)
    unique!(selected_pages)
    @show length(gallery) selected_pages pages

    n = 0
    @time "copy to src" for (root, dirs, files) in walkdir(SRC_DIR)
        prefix = replace(root, SRC_DIR => WRK_DIR)
        foreach(dir -> mkpath(joinpath(WRK_DIR, dir)), dirs)
        for file in files
            _, ext = splitext(file)
            (ext == ".md" && file ∉ selected_pages) && continue
            src = joinpath(root, file)
            dst = joinpath(prefix, file)
            if debug
                endswith(root, r"RecipesBase|RecipesPipeline|UnitfulExt|GraphRecipes|StatsPlots") && continue
                println('\t', src, " -> ", dst)
            end
            cp(src, dst; force = true)
            n += 1
        end
    end
    @info "copied $n source file(s) to scratch directory `$work_dir`"

    if !debug
        @info "UnitfulExt"
        src_unitfulext = "src/UnitfulExt"
        unitfulext = joinpath(@__DIR__, src_unitfulext)
        notebooks = joinpath(unitfulext, "notebooks")

        execute = true  # set to true for executing notebooks and documenter
        nb = false      # set to true to generate the notebooks
        @time "UnitfulExt" for (root, _, files) in walkdir(unitfulext), file in files
            last(splitext(file)) == ".jl" || continue
            ipath = joinpath(root, file)
            opath = replace(ipath, src_unitfulext => joinpath(work_dir, "generated")) |> splitdir |> first
            Literate.markdown(ipath, opath; documenter = execute)
            nb && Literate.notebook(ipath, notebooks; execute)
        end
    end

    failed = false
    if :make ∈ build_cmds
        ansicolor = Base.get_bool_env("PLOTDOCS_ANSICOLOR", true)
        @info "makedocs ansicolor=$ansicolor"
        try
            @time "makedocs" makedocs(;
                format = Documenter.HTML(;
                    size_threshold = nothing,
                    prettyurls = Base.get_bool_env("CI", false),
                    assets = [joinpath(ASSETS, "favicon.ico"), gallery_assets...],
                    collapselevel = 2,
                    edit_link = BRANCH,
                    ansicolor,
                ),
                root = @__DIR__,
                source = work_dir,
                build = bld_dir,
                # pagesonly = true,  # fails DemoCards, see github.com/JuliaDocs/DemoCards.jl/issues/162
                sitename = "Plots",
                authors = "Thomas Breloff",
                warnonly = true,
                pages,
            )
        catch e
            failed = true
            e isa InterruptException || rethrow()
        end
    end
    failed && return  # don't deploy and post-process on failure

    if :deploy ∈ build_cmds
        @info "gallery callbacks"  # URL redirection for DemoCards-generated gallery
        @time "gallery callbacks" foreach(cb -> cb(), gallery_callbacks)

        @info "post-process gallery html files to remove `rng` in user displayed code in gallery"
        # non-exhaustive list of examples to be fixed:
        # [1, 4, 5, 7:12, 14:21, 25:27, 29:30, 33:34, 36, 38:39, 41, 43, 45:46, 48, 52, 54, 62]
        @time "post-process `rng`" for pkg in packages
            be = packages_backends[pkg]
            prefix = joinpath(BLD_DIR, "gallery", string(be), "generated" * suffix)
            must_fix = needs_rng_fix[pkg]
            for file in Glob.glob("*/index.html", prefix)
                (m = match(r"-ref(\d+)", file)) ≡ nothing && continue
                idx = parse(Int, first(m.captures))
                get(must_fix, idx, false) || continue
                lines = readlines(file; keep = true)
                open(file, "w") do io
                    count, in_code, sub = 0, false, ""
                    for line in lines
                        trailing = if (m = match(r"""<code class="language-julia hljs">.*""", line)) ≢ nothing
                            in_code = true
                            m.match
                        else
                            line
                        end
                        if in_code && occursin("rng", line)
                            line = replace(line, r"rng\s*?,\s*" => "")
                            count += 1
                        end
                        occursin("</code>", trailing) && (in_code = false)
                        write(io, line)
                    end
                    count > 0 && @info "replaced $count `rng` occurrence(s) in $file" maxlog = 10
                    @assert count > 0 "idx=$idx - count=$count - file=$file"
                end
            end
        end

        # post-process files for edit link
        @info "post-process work dir to fix edit link in `html` files"
        @time "post-process work dir" for file in vcat(
                Glob.glob("**/*.html", BLD_DIR),  # NOTE: this does not match BLD_DIR/index.html :/
                Glob.glob("*.html", BLD_DIR),  # I don't understand how Glob works :/
            ) |> unique!
            # @show file
            lines = readlines(file; keep = true)
            any(line -> occursin(joinpath("blob", BRANCH, "docs"), line), lines) || continue
            old = joinpath("blob", BRANCH, "docs", work_dir)
            new = joinpath("blob", BRANCH, "docs", src_dir)
            @info "fixing $file $old -> $new" maxlog = 10
            open(file, "w") do io
                foreach(line -> write(io, replace(line, old => new, joinpath(WRK_DIR) => new)), lines)
            end
        end

        debug && for (rt, dirs, fns) in walkdir(BLD_DIR)
            if length(dirs) > 0
                println("dirs in $rt:")
                foreach(d -> println('\t', joinpath(rt, d)), dirs)
            end
            if length(fns) > 0
                println("files in $rt:")
                foreach(f -> println('\t', joinpath(rt, f)), fns)
            end
        end

        @info "deploydocs"
        repo = "JuliaPlots/Plots.jl"
        @time "deploydocs" withenv("GITHUB_REPOSITORY" => repo) do
            deploydocs(;
                root = @__DIR__,
                target = bld_dir,
                deploy_repo = "github.com/JuliaPlots/PlotDocs.jl",  # see https://documenter.juliadocs.org/stable/man/hosting/#Out-of-repo-deployment
                repo_previews = "github.com/JuliaPlots/PlotDocs.jl",
                push_preview = Base.get_bool_env("PLOTDOCS_PUSH_PREVIEW", false),
                devbranch = BRANCH,
                forcepush = true,
                repo,
            )
        end
    end
    @info "done !"
    return nothing
end

@main
