/*
 * Decompiled with CFR 0.152.
 */
package org.xbill.DNS.hosts;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.InetAddress;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xbill.DNS.Address;
import org.xbill.DNS.Name;
import org.xbill.DNS.TextParseException;

public final class HostsFileParser {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(HostsFileParser.class);
    private final int maxFullCacheFileSizeBytes = Integer.parseInt(System.getProperty("dnsjava.hostsfile.max_size_bytes", "16384"));
    private final Duration fileChangeCheckInterval = Duration.ofMillis(Integer.parseInt(System.getProperty("dnsjava.hostsfile.change_check_interval_ms", "300000")));
    private final Path path;
    private final boolean clearCacheOnChange;
    private Clock clock = Clock.systemUTC();
    private volatile Map<String, InetAddress> hostsCache;
    private Instant lastFileModificationCheckTime = Instant.MIN;
    private Instant lastFileReadTime = Instant.MIN;
    private boolean isEntireFileParsed;
    private boolean hostsFileWarningLogged = false;
    private long hostsFileSizeBytes;

    public HostsFileParser() {
        this(System.getProperty("os.name").contains("Windows") ? Paths.get(System.getenv("SystemRoot"), "\\System32\\drivers\\etc\\hosts") : Paths.get("/etc/hosts", new String[0]), true);
    }

    public HostsFileParser(Path path) {
        this(path, true);
    }

    public HostsFileParser(Path path, boolean clearCacheOnChange) {
        this.path = Objects.requireNonNull(path, "path is required");
        this.clearCacheOnChange = clearCacheOnChange;
        if (Files.isDirectory(path, new LinkOption[0])) {
            throw new IllegalArgumentException("path must be a file");
        }
    }

    public Optional<InetAddress> getAddressForHost(Name name, int type) throws IOException {
        Objects.requireNonNull(name, "name is required");
        if (type != 1 && type != 28) {
            throw new IllegalArgumentException("type can only be A or AAAA");
        }
        this.validateCache();
        InetAddress cachedAddress = this.hostsCache.get(this.key(name, type));
        if (cachedAddress != null) {
            return Optional.of(cachedAddress);
        }
        if (this.isEntireFileParsed) {
            return Optional.empty();
        }
        if (this.hostsFileSizeBytes > (long)this.maxFullCacheFileSizeBytes) {
            this.searchHostsFileForEntry(name, type);
        }
        return Optional.ofNullable(this.hostsCache.get(this.key(name, type)));
    }

    private void parseEntireHostsFile() throws IOException {
        int lineNumber = 0;
        AtomicInteger addressFailures = new AtomicInteger(0);
        AtomicInteger nameFailures = new AtomicInteger(0);
        try (BufferedReader hostsReader = Files.newBufferedReader(this.path, StandardCharsets.UTF_8);){
            String line;
            while ((line = hostsReader.readLine()) != null) {
                LineData lineData;
                if ((lineData = this.parseLine(++lineNumber, line, addressFailures, nameFailures)) == null) continue;
                for (Name name : lineData.names) {
                    InetAddress lineAddress = InetAddress.getByAddress(name.toString(true), lineData.address);
                    this.hostsCache.putIfAbsent(this.key(name, lineData.type), lineAddress);
                }
            }
        }
        if (!(this.hostsFileWarningLogged || addressFailures.get() <= 0 && nameFailures.get() <= 0)) {
            log.warn("Failed to parse entire hosts file {}, address failures={}, name failures={}", new Object[]{this.path, addressFailures.get(), nameFailures});
            this.hostsFileWarningLogged = true;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void searchHostsFileForEntry(Name name, int type) throws IOException {
        int lineNumber = 0;
        AtomicInteger addressFailures = new AtomicInteger(0);
        AtomicInteger nameFailures = new AtomicInteger(0);
        try (BufferedReader hostsReader = Files.newBufferedReader(this.path, StandardCharsets.UTF_8);){
            String line;
            while ((line = hostsReader.readLine()) != null) {
                LineData lineData;
                if ((lineData = this.parseLine(++lineNumber, line, addressFailures, nameFailures)) == null) continue;
                for (Name lineName : lineData.names) {
                    boolean isSearchedEntry = lineName.equals(name);
                    if (!isSearchedEntry || type != lineData.type) continue;
                    InetAddress lineAddress = InetAddress.getByAddress(lineName.toString(true), lineData.address);
                    this.hostsCache.putIfAbsent(this.key(lineName, lineData.type), lineAddress);
                    return;
                }
            }
        }
        if (this.hostsFileWarningLogged) return;
        if (addressFailures.get() <= 0) {
            if (nameFailures.get() <= 0) return;
        }
        log.warn("Failed to find {} in hosts file {}, address failures={}, name failures={}", new Object[]{name, this.path, addressFailures.get(), nameFailures});
        this.hostsFileWarningLogged = true;
    }

    private LineData parseLine(int lineNumber, String line, AtomicInteger addressFailures, AtomicInteger nameFailures) {
        String[] lineTokens = this.getLineTokens(line);
        if (lineTokens.length < 2) {
            return null;
        }
        int lineAddressType = 1;
        byte[] lineAddressBytes = Address.toByteArray(lineTokens[0], 1);
        if (lineAddressBytes == null) {
            lineAddressBytes = Address.toByteArray(lineTokens[0], 2);
            lineAddressType = 28;
        }
        if (lineAddressBytes == null) {
            log.debug("Could not decode address {}, {}#L{}", new Object[]{lineTokens[0], this.path, lineNumber});
            addressFailures.incrementAndGet();
            return null;
        }
        Iterable lineNames = Arrays.stream(lineTokens).skip(1L).map(lineTokenName -> this.safeName((String)lineTokenName, lineNumber, nameFailures)).filter(Objects::nonNull)::iterator;
        return new LineData(lineAddressType, lineAddressBytes, lineNames);
    }

    private Name safeName(String name, int lineNumber, AtomicInteger nameFailures) {
        try {
            return Name.fromString(name, Name.root);
        }
        catch (TextParseException e) {
            log.debug("Could not decode name {}, {}#L{}, skipping", new Object[]{name, this.path, lineNumber});
            nameFailures.incrementAndGet();
            return null;
        }
    }

    private String[] getLineTokens(String line) {
        int commentStart = line.indexOf(35);
        if (commentStart == -1) {
            commentStart = line.length();
        }
        return line.substring(0, commentStart).trim().split("\\s+");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void validateCache() throws IOException {
        if (!this.clearCacheOnChange) {
            if (this.hostsCache == null) {
                HostsFileParser hostsFileParser = this;
                synchronized (hostsFileParser) {
                    if (this.hostsCache == null) {
                        this.readHostsFile();
                    }
                }
            }
            return;
        }
        if (this.lastFileModificationCheckTime.plus(this.fileChangeCheckInterval).isBefore(this.clock.instant())) {
            log.debug("Checked for changes more than 5minutes ago, checking");
            HostsFileParser hostsFileParser = this;
            synchronized (hostsFileParser) {
                if (!this.lastFileModificationCheckTime.plus(this.fileChangeCheckInterval).isBefore(this.clock.instant())) {
                    log.debug("Never mind, check fulfilled in another thread");
                    return;
                }
                this.lastFileModificationCheckTime = this.clock.instant();
                this.readHostsFile();
            }
        }
    }

    private void readHostsFile() throws IOException {
        if (Files.exists(this.path, new LinkOption[0])) {
            Instant fileTime = Files.getLastModifiedTime(this.path, new LinkOption[0]).toInstant();
            if (!this.lastFileReadTime.equals(fileTime)) {
                this.createOrClearCache();
                this.hostsFileSizeBytes = Files.size(this.path);
                if (this.hostsFileSizeBytes <= (long)this.maxFullCacheFileSizeBytes) {
                    this.parseEntireHostsFile();
                    this.isEntireFileParsed = true;
                }
                this.lastFileReadTime = fileTime;
            }
        } else {
            this.createOrClearCache();
        }
    }

    private void createOrClearCache() {
        if (this.hostsCache == null) {
            this.hostsCache = new ConcurrentHashMap<String, InetAddress>();
        } else {
            this.hostsCache.clear();
        }
    }

    private String key(Name name, int type) {
        return name.toString() + '\t' + type;
    }

    int cacheSize() {
        return this.hostsCache == null ? 0 : this.hostsCache.size();
    }

    void setClock(Clock clock) {
        this.clock = clock;
    }

    private static final class LineData {
        final int type;
        final byte[] address;
        final Iterable<? extends Name> names;

        @Generated
        public LineData(int type, byte[] address, Iterable<? extends Name> names) {
            this.type = type;
            this.address = address;
            this.names = names;
        }
    }
}

