import re

import qt

import slicer

from .EditableTreeWidget import EditableTreeWidget

from slicer.i18n import tr as _

# -----------------------------------------------------------------------------
def _map_property(objfunc, name):
    return property(lambda self: getattr(objfunc(self), name),
                    lambda self, value: setattr(objfunc(self), name, value))


# =============================================================================
#
# _ui_EditExtensionMetadataDialog
#
# =============================================================================
class _ui_EditExtensionMetadataDialog:
    # ---------------------------------------------------------------------------
    def __init__(self, parent):
        vLayout = qt.QVBoxLayout(parent)
        formLayout = qt.QFormLayout()

        self.nameEdit = qt.QLineEdit()
        formLayout.addRow(_("Name:"), self.nameEdit)

        self.categoryEdit = qt.QLineEdit()
        formLayout.addRow(_("Category:"), self.categoryEdit)
        self.descriptionEdit = qt.QTextEdit()
        self.descriptionEdit.acceptRichText = False
        formLayout.addRow(_("Description:"), self.descriptionEdit)

        self.contributorsList = EditableTreeWidget()
        self.contributorsList.rootIsDecorated = False
        self.contributorsList.selectionBehavior = qt.QAbstractItemView.SelectRows
        self.contributorsList.selectionMode = qt.QAbstractItemView.ExtendedSelection
        self.contributorsList.setHeaderLabels([_("Name"), _("Organization")])
        formLayout.addRow(_("Contributors:"), self.contributorsList)

        vLayout.addLayout(formLayout)
        vLayout.addStretch(1)

        self.buttonBox = qt.QDialogButtonBox()
        self.buttonBox.setStandardButtons(qt.QDialogButtonBox.Ok |
                                          qt.QDialogButtonBox.Cancel)
        vLayout.addWidget(self.buttonBox)


# =============================================================================
#
# EditExtensionMetadataDialog
#
# =============================================================================
class EditExtensionMetadataDialog:
    project = _map_property(lambda self: self.ui.nameEdit, "text")
    category = _map_property(lambda self: self.ui.categoryEdit, "text")
    description = _map_property(lambda self: self.ui.descriptionEdit, "plainText")
    # ---------------------------------------------------------------------------
    def __init__(self, parent):
        self.dialog = qt.QDialog(parent)
        self.ui = _ui_EditExtensionMetadataDialog(self.dialog)

        self.ui.buttonBox.connect("accepted()", self.accept)
        self.ui.buttonBox.connect("rejected()", self.dialog, "reject()")

    # ---------------------------------------------------------------------------
    def accept(self):
        if not len(self.project):
            slicer.util.errorDisplay(_("Extension name may not be empty."), windowTitle=_("Invalid metadata"), parent=self.dialog)
            return

        if not len(self.description):
            slicer.util.errorDisplay(_("Extension description may not be empty."),
                                     windowTitle=_("Invalid metadata"), parent=self.dialog)
            return

        self.dialog.accept()

    # ---------------------------------------------------------------------------
    def exec_(self):
        return self.dialog.exec_()

    # ---------------------------------------------------------------------------
    @property
    def contributors(self):
        result = []
        for row in range(self.ui.contributorsList.itemCount):
            item = self.ui.contributorsList.topLevelItem(row)
            name = item.text(0)
            organization = item.text(1)
            if len(organization):
                result.append(f"{name} ({organization})")
            else:
                result.append(name)
        return ", ".join(result)

    # ---------------------------------------------------------------------------
    @contributors.setter
    def contributors(self, value):
        self.ui.contributorsList.clear()
        for c in re.split(r"(?<=[)])\s*,", value):
            c = c.strip()
            item = qt.QTreeWidgetItem()

            try:
                n = c.index("(")
                item.setText(0, c[:n].strip())
                item.setText(1, c[n + 1 : -1].strip())

            except ValueError:
                qt.qWarning(f"{c}: badly formatted contributor")
                item.setText(0, c)

            self.ui.contributorsList.addItem(item)
