import { IExperimentationService } from '../contracts/IExperimentationService';
import { IExperimentationTelemetry } from '../contracts/IExperimentationTelemetry';
import { IKeyValueStorage } from '../contracts/IKeyValueStorage';
import { IFeatureProvider, FeatureData } from './FeatureProvider/IFeatureProvider';
/**
 * Experimentation service to provide functionality of A/B experiments:
 * - reading flights;
 * - caching current set of flights;
 * - get answer on if flights are enabled.
 */
export declare abstract class ExperimentationServiceBase implements IExperimentationService {
    protected telemetry: IExperimentationTelemetry;
    protected assignmentContextTelemetryPropertyName: string;
    protected telemetryEventName: string;
    protected storageKey?: string | undefined;
    protected storage?: IKeyValueStorage | undefined;
    protected featureProviders?: IFeatureProvider[];
    protected fetchPromise?: Promise<FeatureData[]>;
    protected featuresConsumed: boolean;
    private loadCachePromise;
    readonly initializePromise: Promise<void>;
    private resolveInitialFetchPromise;
    readonly initialFetch: Promise<void>;
    private cachedTelemetryEvents;
    private _features;
    private get features();
    private set features(value);
    constructor(telemetry: IExperimentationTelemetry, assignmentContextTelemetryPropertyName: string, telemetryEventName: string, storageKey?: string | undefined, storage?: IKeyValueStorage | undefined);
    /**
     * Gets all the features from the provider sources (not cache).
     * It returns these features and will also update the providers to have the latest features cached.
     */
    protected getFeaturesAsync(overrideInMemoryFeatures?: boolean): Promise<FeatureData>;
    /**
     *
     * @param featureResults The feature results obtained from all the feature providers.
     */
    protected updateFeatures(featureResults: FeatureData[], overrideInMemoryFeatures?: boolean): void;
    private loadCachedFeatureData;
    /**
     * Returns a value indicating whether the given flight is enabled.
     * It uses the in-memory cache.
     * @param flight The flight to check.
     */
    isFlightEnabled(flight: string): boolean;
    /**
     * Returns a value indicating whether the given flight is enabled.
     * It uses the values currently on cache.
     * @param flight The flight to check.
     */
    isCachedFlightEnabled(flight: string): Promise<boolean>;
    /**
     * Returns a value indicating whether the given flight is enabled.
     * It re-fetches values from the server.
     * @param flight the flight to check.
     */
    isFlightEnabledAsync(flight: string): Promise<boolean>;
    /**
     * Returns the value of the treatment variable, or undefined if not found.
     * It uses the values currently in memory, so the experimentation service
     * must be initialized before calling.
     * @param config name of the config to check.
     * @param name name of the treatment variable.
     */
    getTreatmentVariable<T extends boolean | number | string>(configId: string, name: string): T | undefined;
    /**
     * Returns the value of the treatment variable, or undefined if not found.
     * It re-fetches values from the server. If checkCache is set to true and the value exists
     * in the cache, the Treatment Assignment Service is not called.
     * @param config name of the config to check.
     * @param name name of the treatment variable.
     * @param checkCache check the cache for the variable before calling the TAS.
     */
    getTreatmentVariableAsync<T extends boolean | number | string>(configId: string, name: string, checkCache?: boolean): Promise<T | undefined>;
    private PostEventToTelemetry;
    protected invokeInit(): void;
    /**
     * Method to do any post-base constructor calls.
     * Consider this a constructor for the derived classes.
     * Can be used to initialize the Feature Providers.
     * No async calls should be done here.
     */
    protected abstract init(): void;
    protected addFeatureProvider(...providers: IFeatureProvider[]): void;
}
